package nm

import (
	"bytes"
	"encoding/json"
	"example.com/m/conf"
	"example.com/m/db"
	"example.com/m/log"
	"example.com/m/models"
	"example.com/m/operate"
	"example.com/m/utils"
	"fmt"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/crypto"
	"github.com/golang/groupcache/lru"
	baseV1 "github.com/odysseus/odysseus-protocol/gen/proto/go/base/v1"
	nodemanagerV2 "github.com/odysseus/odysseus-protocol/gen/proto/go/nodemanager/v2"
	"io"
	"math/rand"
	"mime/multipart"
	"net/http"
	"net/url"
	"strconv"
	"strings"
	"sync"
	"time"
)

type TaskWorker struct {
	Wg                                    *sync.WaitGroup
	LruCache                              *lru.Cache
	DockerOp                              *operate.DockerOp
	CmdOp                                 *operate.Command
	TaskMsg                               chan *nodemanagerV2.PushTaskMessage
	IsExecAiTask                          bool
	IsExecStandardTask                    bool
	ExecTaskIdIsFinished                  *sync.Map
	lastExecTaskId, lastExecTaskImageName string
	lastExecTaskStartTime                 time.Time
}

type TaskOp struct {
	taskMsg             *nodemanagerV2.PushTaskMessage
	taskCmd             *models.TaskCmd
	taskExecResult      *models.TaskResult
	taskParam           *models.TaskParam
	httpClient          *http.Client
	request             *http.Request
	waitRunningTicker   *time.Ticker
	waitReqTicker       *time.Ticker
	startBeforeTaskTime time.Time
}

func NewTaskWorker(op *operate.DockerOp) *TaskWorker {
	return &TaskWorker{
		Wg:                   &sync.WaitGroup{},
		LruCache:             lru.New(100),
		DockerOp:             op,
		TaskMsg:              make(chan *nodemanagerV2.PushTaskMessage, 0),
		IsExecAiTask:         false,
		ExecTaskIdIsFinished: &sync.Map{},
	}
}

func (t *TaskWorker) DistributionTaskWorker(runCount int) {
	for i := 0; i < runCount; i++ {
		go func(t *TaskWorker) {
			for {
				select {
				case taskMsg := <-t.TaskMsg:
					{
						switch taskMsg.TaskKind {
						case baseV1.TaskKind_SystemTask:
							{
								//command := operate.GetSystemCommand(taskMsg.TaskCmd, taskMsg.TaskParam, taskMsg.TaskId+".sh")
								t.SystemTaskHandler(taskMsg)
							}
						case baseV1.TaskKind_ComputeTask:
							{
								t.ComputeTaskHandler(taskMsg)
							}
						case baseV1.TaskKind_CustomTask:
							{
								t.CustomTaskHandler(taskMsg)
							}
						case baseV1.TaskKind_StandardTask:
							{
								t.ComputeTaskHandler(taskMsg)
							}
						}
					}
				}
			}
		}(t)
	}
}

func (t *TaskWorker) GetMinerSign(msg *nodemanagerV2.PushTaskMessage, taskResult []byte) ([]byte, []byte, []byte) {
	reqHash := crypto.Keccak256Hash(msg.TaskParam)
	respHash := crypto.Keccak256Hash(taskResult)
	signHash := crypto.Keccak256Hash(bytes.NewBufferString(msg.TaskId).Bytes(), reqHash.Bytes(), respHash.Bytes())
	log.WithField("hash", signHash.String()).Info("Miner sign result")
	sign, err := crypto.Sign(signHash.Bytes(), conf.GetConfig().SignPrivateKey)
	if err != nil {
		log.Error("custom task handler")
	}
	log.Info("Miner sign:", common.Bytes2Hex(sign))
	return reqHash.Bytes(), respHash.Bytes(), sign
}

func (t *TaskWorker) SystemTaskHandler(taskMsg *nodemanagerV2.PushTaskMessage) {
	defer t.Wg.Done()

	log.Info("received systemTask--------------------------------")
}

func (t *TaskWorker) CustomTaskHandler(taskMsg *nodemanagerV2.PushTaskMessage) {
	defer t.Wg.Done()
	_, err := t.DockerOp.PsImages()
	if err != nil {
		log.Error("custom task handler docker op ps images failed: ", err)
		return
	}
	log.Info("received customTask--------------------------------")
}

func (t *TaskWorker) ComputeTaskHandler(taskMsg *nodemanagerV2.PushTaskMessage) {
	defer t.Wg.Done()
	t.lastExecTaskStartTime = time.Now()
	t.checkLastTaskExecStatus(taskMsg)
	log.Info("check last task exec status successful")
	bodyByte := utils.EncodeJsonEscapeHTML([]string{"https://replicate.delivery/pbxt/KW7CdCusXMkkOs9bbCGYsInC8EUxlj3yBLxvfW9Fs9FFMZUL/MTk4MTczMTkzNzI1Mjg5NjYy.webp"})
	header := make(map[string][]string, 0)
	header["Content-Type"] = []string{"application/json"}
	headerByte, _ := json.Marshal(header)
	taskOp := &TaskOp{
		taskMsg: taskMsg,
		taskCmd: &models.TaskCmd{},
		taskExecResult: &models.TaskResult{
			TaskHttpStatusCode: 200,
			TaskRespBody:       bodyByte,
			TaskHttpHeaders:    headerByte,
			TaskIsSuccess:      true,
			TaskExecTime:       0,
			TaskExecError:      "",
		},
		taskParam:           &models.TaskParam{},
		httpClient:          &http.Client{},
		request:             &http.Request{},
		waitRunningTicker:   time.NewTicker(time.Millisecond),
		waitReqTicker:       time.NewTicker(time.Millisecond),
		startBeforeTaskTime: time.Now(),
	}
	t.LruCache.Add(taskMsg.TaskId, taskOp.taskExecResult)
	err := json.Unmarshal(bytes.NewBufferString(taskMsg.TaskCmd).Bytes(), taskOp.taskCmd)
	if err != nil {
		log.Errorf("failed to unmarshal task cmd: %s", err.Error())
		taskOp.taskExecResult.TaskExecError = fmt.Sprintf("%s,%s", "failed to unmarshal task cmd: %s", err.Error())
		t.ExecTaskIdIsFinished.Store(taskMsg.TaskId, true)
		return
	}
	taskOp.taskCmd.ImageName = fmt.Sprintf("%s-%s", taskOp.taskCmd.ImageName, conf.GetConfig().OpSys)
	model, err := db.GetModel(taskOp.taskCmd.ImageName)
	if err != nil {
		log.Errorf("failed to unmarshal task cmd: %s", err.Error())
		taskOp.taskExecResult.TaskExecError = fmt.Sprintf("%s,%s", "Not found location model info: %s", err.Error())
		t.ExecTaskIdIsFinished.Store(taskMsg.TaskId, true)
		return
	}
	model.LastWorkTime = time.Now().Unix()
	model.TotalRunCount++
	log.Info("received task cmd :", taskOp.taskCmd)
	log.WithField("t.lastExecTaskImageName", t.lastExecTaskImageName).WithField("newTaskImageName", taskOp.taskCmd.ImageName).Info("task image info")
	if taskMsg.TaskKind != baseV1.TaskKind_StandardTask && conf.GetConfig().IsStopLastContainer {
		t.checkIsStopContainer(taskOp.taskCmd)
	}
	log.Info("check is stop container finished")
	err = json.Unmarshal(taskMsg.TaskParam, taskOp.taskParam)
	if err != nil {
		log.WithField("err", err).Error("Error unmarshalling task parameter")
		taskOp.taskExecResult.TaskExecError = fmt.Sprintf("%s,%s", "Error unmarshalling task parameter", err.Error())
		t.ExecTaskIdIsFinished.Store(taskMsg.TaskId, true)
		return
	}
	time.Sleep(time.Millisecond * 100)
	//running, _ := t.foundImageIsRunning(taskOp.taskCmd.ImageName)
	//if !running {
	//	taskOp.taskCmd.DockerCmd.HostIp = models.ZeroHost
	//	taskOp.taskCmd.DockerCmd.HostPort = t.getExternalPort()
	//	info := GetHardwareInfo()
	//	if info == nil {
	//		log.Error("Error getting hardware info")
	//		taskOp.taskExecResult.TaskExecError = fmt.Sprintf("%s", "Error getting hardware info")
	//		t.ExecTaskIdIsFinished.Store(taskMsg.TaskId, true)
	//		return
	//	}
	//	containerId, gpuSeq, err := t.DockerOp.CreateAndStartContainer(info, model, taskOp.taskCmd.DockerCmd)
	//	if err != nil {
	//		log.Errorf("Create and start container failed: %s", err.Error())
	//		taskOp.taskExecResult.TaskExecError = fmt.Sprintf("%s,%s", "Create and start container failed", err.Error())
	//		t.ExecTaskIdIsFinished.Store(taskMsg.TaskId, true)
	//		return
	//	}
	//	model.GpuSeq = gpuSeq
	//	log.Info("Started container with ID:", containerId)
	//}
	//if err = taskOp.waitContainerRunning(t, taskOp.taskCmd.ImageName, uint16(taskOp.taskCmd.DockerCmd.ContainerPort)); err != nil {
	//	taskOp.taskExecResult.TaskExecError = fmt.Sprintf("%s", err.Error())
	//	t.ExecTaskIdIsFinished.Store(taskMsg.TaskId, true)
	//	return
	//}
	//if err = taskOp.waitReqContainerOk(t.DockerOp); err != nil {
	//	taskOp.taskExecResult.TaskExecError = fmt.Sprintf("%s", err.Error())
	//	t.ExecTaskIdIsFinished.Store(taskMsg.TaskId, true)
	//	return
	//}
	endAfterTaskTime := time.Since(taskOp.startBeforeTaskTime)
	taskOp.taskExecResult.TaskExecTime = endAfterTaskTime.Microseconds()
	log.WithField("time", endAfterTaskTime.Seconds()).WithField("taskId", taskMsg.TaskId).Info("Exec task end (second is units) :")
	if taskMsg.TaskKind == baseV1.TaskKind_ComputeTask {
		t.IsExecAiTask = false
	} else if taskMsg.TaskKind == baseV1.TaskKind_StandardTask {
		t.IsExecStandardTask = false
	}
	model.EstimatExeTime = int32(endAfterTaskTime.Seconds())
	t.ExecTaskIdIsFinished.Store(taskMsg.TaskId, true)
	_ = db.PutModel(taskOp.taskCmd.ImageName, model)
	//log.WithField("result", taskExecResult).Info("lru cache storage task result")
	log.Info("----------------------Compute task exec done--------------------------------")
}

func (t *TaskWorker) GetAckResp(taskMsg *nodemanagerV2.PushTaskMessage) (isCanExecute bool, bootUpTime, queueWaitTime, executeTime int64, imageName string) {
	if t.IsExecStandardTask {
		isCanExecute = true
		return
	}
	taskCmd := &models.TaskCmd{}
	err := json.Unmarshal(bytes.NewBufferString(taskMsg.TaskCmd).Bytes(), taskCmd)
	if err != nil {
		log.Errorf("failed to unmarshal task cmd: %s", err.Error())
		return
	}
	taskCmd.ImageName = fmt.Sprintf("%s-%s", taskCmd.ImageName, conf.GetConfig().OpSys)
	value, ok := t.ExecTaskIdIsFinished.Load(t.lastExecTaskId)
	if !ok {
		log.WithField("task id", t.lastExecTaskId).Warn("task exec is not finished")
	} else {
		isSuccess := value.(bool)
		log.WithField("isSuccess", isSuccess).Info("Task exec info")
		if !isSuccess && !t.lastExecTaskStartTime.IsZero() {
			lastTaskImageInfo, err := db.GetModel(t.lastExecTaskImageName)
			if err != nil {
				return false, 0, 0, 0, ""
			}
			since := time.Since(t.lastExecTaskStartTime)
			queueWaitTime = int64(lastTaskImageInfo.EstimatExeTime - int32(since.Seconds()))
			if queueWaitTime < 0 {
				queueWaitTime = int64(lastTaskImageInfo.EstimatExeTime)
			}
		}
	}
	//if t.foundTaskImage(taskCmd) == "" {
	//	log.WithField("imageName", taskCmd.ImageName).Error("The image is not found")
	//	return
	//}
	//running, _ := t.foundImageIsRunning(taskCmd.ImageName)
	//if !running {
	//
	//}
	log.Info("found task image finished")
	isCanExecute = true
	modelInfo, err := db.GetModel(taskCmd.ImageName)
	if err != nil {
		return false, 0, 0, 0, ""
	}
	if modelInfo != nil {
		bootUpTime = modelInfo.StartUpTime
		executeTime = int64(modelInfo.EstimatExeTime)
	}
	imageName = taskCmd.ImageName
	return
}

func (t *TaskWorker) foundTaskImage(taskCmd *models.TaskCmd) (imageId string) {
	images, err := t.DockerOp.PsImages()
	if err != nil {
		log.Error("Ps images failed:", err)
		imageId = ""
		return
	}
	foundImageName := fmt.Sprintf("%s-%s", taskCmd.ImageName, conf.GetConfig().OpSys)
	isFound := false
	for _, image := range images {
		if isFound {
			break
		}
		for _, tag := range image.RepoTags {
			if tag == foundImageName {
				imageId = image.ID
				isFound = true
				log.Info("The image found success:", image.ID)
				break
			}
		}
	}
	return
}

func (t *TaskWorker) foundImageIsRunning(imageName string) (bool, string) {
	containers := t.DockerOp.ListContainer()
	for _, container := range containers {
		if container.Image == imageName && container.State == "running" {
			networks := container.NetworkSettings.Networks
			ip := ""
			for _, endPoint := range networks {
				ip = endPoint.IPAddress
				log.Warn("Container network ip:", ip)
			}
			return true, ip
		}
	}
	return false, ""
}

func (t *TaskWorker) checkLastTaskExecStatus(taskMsg *nodemanagerV2.PushTaskMessage) {
	if taskMsg.TaskKind == baseV1.TaskKind_ComputeTask {
		t.IsExecAiTask = true
		if t.IsExecStandardTask {
			//todo: 停止标准任务容器
			//containers := t.DockerOp.ListContainer()
			//for _, container := range containers {
			//	if container.Image == taskCmd.ImageName && container.State == "running" {
			//		t.DockerOp.StopContainer(container.ID)
			//	}
			//}
			t.IsExecStandardTask = false
		}
	} else if taskMsg.TaskKind == baseV1.TaskKind_StandardTask {
		t.IsExecStandardTask = true
	}
	if t.lastExecTaskId != taskMsg.TaskId {
		now := time.Now()
		for {
			since := time.Since(now)
			if int64(since.Seconds()) > conf.GetConfig().WaitLastTaskExecTime {
				log.WithField("taskId", t.lastExecTaskId).Info("Waiting for last task execution ending")
				t.lastExecTaskId = taskMsg.TaskId
				break
			}
			if t.lastExecTaskId == "" {
				t.lastExecTaskId = taskMsg.TaskId
				break
			}
			value, ok := t.ExecTaskIdIsFinished.Load(t.lastExecTaskId)
			//log.WithField("isSuccess", value).Info("Task id exec info")
			if !ok {
				//log.WithField("task id", t.lastExecTaskId).Warn("task exec is not finished")
				continue
			}
			isSuccess := value.(bool)
			if isSuccess {
				t.lastExecTaskId = taskMsg.TaskId
				log.WithField("taskId", t.lastExecTaskId).Info("Task exec success")
				break
			}
		}
	}
}

func (t *TaskWorker) checkIsStopContainer(taskCmd *models.TaskCmd) {
	if t.lastExecTaskImageName != "" && t.lastExecTaskImageName != taskCmd.ImageName {
		//todo: 停止标准任务容器
		containers := t.DockerOp.ListContainer()
		for _, container := range containers {
			split := strings.Split(container.Image, ":")
			if len(split) == 1 {
				container.Image = fmt.Sprintf("%s:%s", container.Image, "latest")
			}
			log.WithField("containerImageName", container.Image).WithField("t.lastExecTaskImageName", t.lastExecTaskImageName).Info("match image")
			if container.Image == t.lastExecTaskImageName && container.State == "running" {
				t.DockerOp.StopContainer(container.ID)
				log.WithField("Image name", container.Image).Info("Stopping container")
				//t.DockerOp.RunningImages[t.lastExecTaskImageName] = false
				break
			}
		}
		t.lastExecTaskImageName = taskCmd.ImageName
	} else {
		t.lastExecTaskImageName = taskCmd.ImageName
	}
}

func (t *TaskWorker) getExternalPort() (externalPort string) {
	for {
		// 设置种子以确保每次运行时生成不同的随机数序列
		rand.Seed(time.Now().UnixNano())
		// 生成一个介于 0 和 100 之间的随机整数
		externalPortInt := rand.Int63n(10001) + 10000
		log.WithField("externalPortInt", externalPortInt).Info("DockerOp UsedExternalPort :", t.DockerOp.UsedExternalPort[externalPortInt])
		if t.DockerOp.UsedExternalPort[externalPortInt] {
			continue
		}
		externalPort = strconv.FormatInt(externalPortInt, 10)
		break
	}
	return
}

func (op *TaskOp) uploadOSS(taskId string, queries string, decodedImage []byte, suffix string) (string, error) {
	var requestBody bytes.Buffer
	writer := multipart.NewWriter(&requestBody)
	// 创建文件表单字段
	fileField, err := writer.CreateFormFile("file", fmt.Sprintf("%s.%s", taskId, suffix))
	if err != nil {
		log.WithError(err).Error("Error creating form file")
		return "", err
	}
	_, err = io.Copy(fileField, bytes.NewReader(decodedImage))
	//_, err = io.Copy(fileField, strings.NewReader(base64Image))
	if err != nil {
		log.WithError(err).Error("Error copying file contents")
		return "", err
	}
	// 关闭 multipart writer
	err = writer.Close()
	if err != nil {
		log.WithError(err).Error("Error closing writer")
		return "", err
	}
	ossUrl := fmt.Sprintf("%s?%s", conf.GetConfig().OssUrl, queries)
	request, err := http.NewRequest("POST", ossUrl, &requestBody)
	if err != nil {
		return "", err
	}
	request.Header.Set("Content-Type", writer.FormDataContentType())
	response, err := op.httpClient.Do(request)
	if err != nil {
		log.WithError(err).Error("Error request oss failed")
		return "", err
	}
	ossRespBody, err := io.ReadAll(response.Body)
	if err != nil {
		log.WithError(err).Error("Error read oss resp body failed")
		return "", err
	}
	log.WithField("res", string(ossRespBody)).Info("file cache resp body")
	fileCacheRes := &models.FileCacheResult{}
	err = json.Unmarshal(ossRespBody, fileCacheRes)
	if err != nil {
		log.WithError(err).Error("Json unmarshal file cache result failed")
		return "", err
	}
	log.WithField("code", fileCacheRes.Code).WithField("msg", fileCacheRes.Msg).WithField("data", fileCacheRes.Data).Info("file cache result")
	if fileCacheRes.Code == http.StatusOK && fileCacheRes.Data != "" {
		_, err := url.Parse(fileCacheRes.Data)
		if err != nil {
			log.WithError(err).Error("url parse file cache data error")
			return "", err
		}
		return fileCacheRes.Data, nil
	}
	return "", err
}

func (op *TaskOp) getFileCache(respStr string, dockerOp *operate.DockerOp) (string, error) {
	isBase64, decodeByte, respFormat, suffix := utils.IsBase64ImageStr(respStr)
	log.WithField("isBase64", isBase64).Info("resp str info")
	if isBase64 {
		log.WithField("taskId", op.taskMsg.TaskId).WithField("format", respFormat).WithField("suffix", suffix).Info("Parse container resp")
		queryString := utils.MatchFileCacheQueryString(op.taskParam.Headers, op.taskCmd.ImageName, respFormat)
		ossUri, err := op.uploadOSS(op.taskMsg.TaskId, queryString, decodeByte, suffix)
		if err != nil || ossUri == "" {
			log.WithError(err).Error("upload image into file cache failed")
			return "", err
		}
		log.WithField("uri", ossUri).Info("upload image OSS  successful")
		return ossUri, nil
	}
	return "", nil
}

func (op *TaskOp) checkContainerHealthy(internalIp string, internalPort uint16) (bool, error) {
	healthCheckUrl := fmt.Sprintf("http://%s:%d%s", internalIp, internalPort, models.HealthCheckAPI)
	healthyCheckResp, err := op.httpClient.Get(healthCheckUrl)
	if err != nil {
		log.Warn("Request container healthy failed: %s", err.Error())
		return false, nil
	}
	if healthyCheckResp.StatusCode == http.StatusNotFound {
		return true, nil
	}
	body, err := io.ReadAll(healthyCheckResp.Body)
	m := &models.HealthyCheck{}
	err = json.Unmarshal(body, m)
	if err != nil {
		log.Errorf("Json unmarshal container healthy body failed: %s", err.Error())
		return false, fmt.Errorf("%s,%s", "Json unmarshal container healthy body failed", err.Error())
	}
	if m.Status != models.READY {
		log.Warn("The container is not ready")
		return false, nil
	}
	return true, nil
}

func (op *TaskOp) waitContainerRunning(handler *TaskWorker, imageName string, internalPort uint16) error {
	maxExecTime := op.GetMaxExecTime()
	log.WithField("maxExecTime", maxExecTime).Info("Waiting for container running", imageName)
	defer op.waitRunningTicker.Stop()
	for {
		select {
		case <-op.waitRunningTicker.C:
			op.waitRunningTicker = time.NewTicker(time.Second * models.DefaultTaskTimer)
			if int64(time.Since(op.startBeforeTaskTime).Seconds()) > maxExecTime-50 {
				log.Errorf("%s", "The maximum execution time for this task has been exceeded")
				return fmt.Errorf("%s", "The maximum execution time for this task has been exceeded")
			}
			running, internalIp := handler.foundImageIsRunning(imageName)
			if !running {
				continue
			}
			if isMatch := strings.HasPrefix(op.taskCmd.ImageName, conf.GetConfig().ReplicateImageNameSuffix); isMatch {
				if isReqSuccess, err := op.checkContainerHealthy(internalIp, internalPort); err != nil {
					log.WithField("err", err).Errorf("check container healthy failed")
					return fmt.Errorf("%s-%s", "check container healthy failed", err.Error())
				} else if !isReqSuccess {

					continue
				}
			}
			op.taskCmd.ApiUrl = fmt.Sprintf("http://%s:%d%s", internalIp, internalPort, op.taskCmd.ApiUrl)
			log.Info("Container ports:", internalPort)
			log.WithField("ApiUrl", op.taskCmd.ApiUrl).Info("The image is not  running")
			return nil
		}
	}
}

func (op *TaskOp) validateWebHook() error {
	for key, value := range op.taskParam.Headers {
		if key == models.Prefer {
			if value[0] == models.Async {
				op.request.Header.Set(models.Prefer, models.Async)
				m := &models.ContainerRequest{}
				err := json.Unmarshal(op.taskParam.Body, m)
				if err != nil {
					log.WithError(err).Error("json unmarshal task body failed")
					return fmt.Errorf("%s,%s", "Json unmarshal task body failed", err.Error())
				}
				if m.WebHook == "" {
					log.Error("Request webhook is nil")
					return fmt.Errorf("%s", "Request webhook is nil")
				} else {
					_, err := url.Parse(m.WebHook)
					if err != nil {
						log.WithError(err).Error("web hook url parse failed")
						return fmt.Errorf("%s,%s", "Web hook url parse failed", err.Error())
					}
				}
				break
			}
		}
	}
	return nil
}

func (op *TaskOp) waitReqContainerOk(dockerOp *operate.DockerOp) error {
	maxExecTime := op.GetMaxExecTime()
	log.WithField("maxExecTime", maxExecTime).Info("Waiting for request container success")
	var err error
	defer op.waitReqTicker.Stop()
	for {
		select {
		case <-op.waitReqTicker.C:
			op.waitReqTicker = time.NewTicker(time.Second * models.DefaultTaskTimer)
			if int64(time.Since(op.startBeforeTaskTime).Seconds()) > maxExecTime-50 {
				log.Errorf("%s", "The maximum execution time for this task has been exceeded")
				return fmt.Errorf("%s", "The maximum execution time for this task has been exceeded")
			}
			reqContainerBody := bytes.NewReader(op.taskParam.Body)
			if len(op.taskParam.Queries) > 0 {
				queryString := utils.MatchContainerQueryString(op.taskParam.Queries)
				op.taskCmd.ApiUrl = fmt.Sprintf("%s?%s", op.taskCmd.ApiUrl, queryString)
				log.WithField("ApiUrl", op.taskCmd.ApiUrl).Info("The task param query str not empty")
			}
			op.request, err = http.NewRequest("POST", op.taskCmd.ApiUrl, reqContainerBody)
			if err != nil {
				log.WithField("error:", err).Error("New container request failed")
				return fmt.Errorf("%s,%s", "Http client new container request failed", err.Error())
			}
			op.request.Header.Set("Content-Type", "application/json")
			if err = op.validateWebHook(); err != nil {
				return fmt.Errorf("%s", err.Error())
			}
			post, err := op.httpClient.Do(op.request)
			if err != nil {
				log.WithField("error:", err).Error("Http client post request container failed")
				return fmt.Errorf("%s,%s", "Http client post request container failed", err.Error())
			}
			log.WithField("StatusCode", post.StatusCode).WithField("taskId", op.taskMsg.TaskId).Info("Exec task result")
			if post.StatusCode == http.StatusOK {
				op.taskExecResult.TaskHttpStatusCode = http.StatusOK
				readBody, err := io.ReadAll(post.Body)
				if err != nil {
					log.Errorf("%s,%s,Container Http Code:%d", "Read container body failed", err.Error(), post.StatusCode)
					return fmt.Errorf("%s,%s,Container Http Code:%d", "Read container body failed", err.Error(), post.StatusCode)
				}
				if op.taskMsg.TaskKind != baseV1.TaskKind_StandardTask {
					isUseFileCache := true
					isUseRedirect := false
					for key, value := range op.taskParam.Headers {
						log.WithField("key", key).WithField("val", value).Debug("Headers Info")
						if key == models.UseRedirect {
							log.WithField("UseRedirect", value[0]).Info("Headers info")
							if value[0] == "true" {
								isUseRedirect = true
								isUseFileCache = true
							}
						}
						if key == models.UseFileCache {
							log.WithField("UseFileCache", value[0]).Info("Headers info")
							if value[0] == "false" {
								isUseFileCache = false
								break
							}
						}
					}
					log.WithField("isUseRedirect", isUseRedirect).Info("is use redirect")
					log.WithField("isUseFileCache", isUseFileCache).Info("is use file cache")
					if readBody != nil {
						data := parseData(readBody)
						if data != nil {
							isSuccess := false
							switch v := data.(type) {
							case [][]string:
								{
									res := data.([][]string)
									log.Info("data is [][]string type")
									apiRes := make([][]string, 1)
									for _, innerSlice := range res {
										apiResOneArr := make([]string, 0)
										for _, respStr := range innerSlice {
											if respStr == "" || respStr == "null" {
												continue
											}
											if !isUseFileCache {
												apiResOneArr = append(apiResOneArr, respStr)
												isSuccess = true
												continue
											}
											ossUri, err := op.getFileCache(respStr, dockerOp)
											if err != nil || ossUri == "" {
												if err != nil {
													op.taskExecResult.TaskExecError = fmt.Sprintf("%s,%s", "Get file cache uri failed", err)
												}
												if ossUri == "" {
													apiResOneArr = append(apiResOneArr, respStr)
													isSuccess = true
												}
												continue
											}
											if isUseRedirect && ossUri != "" && len(res) == 1 && len(innerSlice) == 1 {
												op.taskExecResult.TaskHttpStatusCode = models.RedirectCode
												apiResBody := utils.EncodeJsonEscapeHTML(ossUri)
												op.taskExecResult.TaskRespBody = apiResBody
												post.Header.Set("Location", ossUri)
												isSuccess = true
												break
											} else {
												apiResOneArr = append(apiResOneArr, ossUri)
												isSuccess = true
											}
										}
										apiRes = append(apiRes, apiResOneArr)
									}
									if !isSuccess {
										return fmt.Errorf("%s-%s", "Container output is nil", string(readBody))
									}
									if !isUseRedirect {
										apiResBody := utils.EncodeJsonEscapeHTML(apiRes)
										op.taskExecResult.TaskRespBody = apiResBody
									}
								}
							case []string:
								{
									res := data.([]string)
									log.Info("data is []string type")
									apiRes := make([]string, 0)
									for _, respStr := range res {
										if respStr == "" || respStr == "null" {
											continue
										}
										if !isUseFileCache {
											apiRes = append(apiRes, respStr)
											isSuccess = true
											continue
										}
										ossUri, err := op.getFileCache(respStr, dockerOp)
										if err != nil || ossUri == "" {
											if err != nil {
												op.taskExecResult.TaskExecError = fmt.Sprintf("%s,%s", "Get file cache uri failed", err)
											}
											if ossUri == "" {
												apiRes = append(apiRes, respStr)
												isSuccess = true
											}
											continue
										}
										if isUseRedirect && ossUri != "" && len(res) == 1 {
											op.taskExecResult.TaskHttpStatusCode = models.RedirectCode
											post.Header.Set("Location", ossUri)
											apiResBody := utils.EncodeJsonEscapeHTML(ossUri)
											op.taskExecResult.TaskRespBody = apiResBody
											isSuccess = true
											break
										} else {
											apiRes = append(apiRes, ossUri)
											isSuccess = true
										}
									}
									if !isSuccess {
										return fmt.Errorf("%s-%s", "Container output is nil", string(readBody))
									}
									if !isUseRedirect {
										apiResBody := utils.EncodeJsonEscapeHTML(apiRes)
										op.taskExecResult.TaskRespBody = apiResBody
									}
								}
							case string:
								{
									resStr := data.(string)
									log.Info("data is string type")
									resArr := []string{resStr}
									apiRes := make([]string, 0)
									for _, respStr := range resArr {
										if respStr == "" || respStr == "null" {
											continue
										}
										if !isUseFileCache {
											apiRes = append(apiRes, respStr)
											isSuccess = true
											continue
										}
										ossUri, err := op.getFileCache(respStr, dockerOp)
										if err != nil || ossUri == "" {
											if err != nil {
												op.taskExecResult.TaskExecError = fmt.Sprintf("%s,%s", "Get file cache uri failed", err)
											}
											if ossUri == "" {
												apiRes = append(apiRes, respStr)
												isSuccess = true
											}
											continue
										}
										if isUseRedirect && ossUri != "" && len(resArr) == 1 {
											op.taskExecResult.TaskHttpStatusCode = models.RedirectCode
											post.Header.Set("Location", ossUri)
											apiResBody := utils.EncodeJsonEscapeHTML(ossUri)
											op.taskExecResult.TaskRespBody = apiResBody
											isSuccess = true
											break
										} else {
											apiRes = append(apiRes, ossUri)
											isSuccess = true
										}
									}
									if !isSuccess {
										return fmt.Errorf("%s-%s", "Container output is nil", string(readBody))
									}
									if !isUseRedirect {
										apiResBody := utils.EncodeJsonEscapeHTML(apiRes)
										op.taskExecResult.TaskRespBody = apiResBody
									}
								}
							default:
								log.Error("data is unknown type", v)
								return fmt.Errorf("%s", "Container resp data is unknown type")
							}
						} else {
							log.Error("Container resp output is nil")
							op.taskExecResult.TaskRespBody = readBody
							return fmt.Errorf("%s", "Container resp output is nil")
						}
					}
				}
				if op.taskMsg.TaskKind == baseV1.TaskKind_StandardTask {
					op.taskExecResult.TaskRespBody = readBody
				}
				headers, err := json.Marshal(post.Header)
				if err != nil {
					log.WithError(err).Error("JSON marshal container header failed")
					return fmt.Errorf("%s,%s", "JSON marshal container header failed", err.Error())
				}
				log.WithField("headers", post.Header).Info("return task http headers")
				op.taskExecResult.TaskHttpHeaders = headers
				op.taskExecResult.TaskIsSuccess = true
			} else {
				op.taskExecResult.TaskHttpStatusCode = int32(post.StatusCode)
				if op.taskExecResult.TaskHttpStatusCode == http.StatusConflict {
					log.Errorf("%s,Container Http Code:%d", "Already running a prediction", post.StatusCode)
					continue
				}
				log.WithField("taskId", op.taskMsg.TaskId).Error("Exec task result is failed")
				if post.Body != nil {
					all, _ := io.ReadAll(post.Body)
					return fmt.Errorf("%s,Container Http Code:%d,body:%s", "Container is exec failed", post.StatusCode, string(all))
				} else {
					return fmt.Errorf("%s,Container Http Code:%d,body:%s", "Container resp body is nil", post.StatusCode, "")
				}
			}
			return nil
		}
	}
}

func (op *TaskOp) GetMaxExecTime() int64 {
	maxExecTimeArr := op.taskParam.Headers[models.MaxExecTime]
	var maxExecTime int64
	if len(maxExecTimeArr) > 0 {
		maxExecTime, _ = strconv.ParseInt(maxExecTimeArr[0], 10, 64)
	}
	if maxExecTime == 0 {
		maxExecTime = models.DefaultMaxExecTime
	}
	return maxExecTime
}

func parseData(readBody []byte) interface{} {
	var m map[string]json.RawMessage
	if err := json.Unmarshal(readBody, &m); err != nil {
		log.WithError(err).Error("Parse json raw message failed")
		return bytes.NewBuffer(readBody).String()
	}
	if m["output"] == nil || len(m["output"]) == 0 {
		log.WithField("output", nil).Warn("The container resp")
		return nil
	}
	var outputTwoArray [][]string
	if err := json.Unmarshal(m["output"], &outputTwoArray); err != nil {
		log.WithField("err", err).Warn("parse two array output filed failed:")
		var outputOneArray []string
		if err := json.Unmarshal(m["output"], &outputOneArray); err != nil {
			log.WithField("err", err).Warn("parse one array output filed failed:")
			var outputString string
			if err := json.Unmarshal(m["output"], &outputString); err != nil {
				log.WithField("err", err).Warn("parse string output filed failed:")
				var audioOutput *models.BarkOutPut
				if err := json.Unmarshal(m["output"], &audioOutput); err != nil {
					log.WithField("err", err).Warn("parse audioOutput output filed failed:")
					swinirOutPutArr := make([]*models.SwinirOutPut, 0)
					if err := json.Unmarshal(m["output"], &audioOutput); err != nil {
						log.WithField("err", err).Warn("parse swinirOutput output filed failed:")
						return nil
					} else {
						res := make([]string, 0)
						for _, out := range swinirOutPutArr {
							res = append(res, out.File)
						}
						return res
					}
				} else {
					return audioOutput.AudioOut
				}
			} else {
				return outputString
			}
		} else {
			return outputOneArray
		}
	} else {
		return outputTwoArray
	}
}
