package main

import (
	"encoding/json"
	"example.com/m/conf"
	"example.com/m/log"
	"example.com/m/nm"
	"example.com/m/utils"
	"fmt"
	"github.com/astaxie/beego"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/crypto"
	"github.com/spf13/cobra"
	"github.com/spf13/viper"
	"io/ioutil"
	"os"
)

var (
	rewardAddr, dockerServer, externalIp, opSys string
)

func init() {
	RootCmd.PersistentFlags().StringVarP(&rewardAddr, "reward", "r", "", "please enter a reward address")
	RootCmd.PersistentFlags().StringVarP(&dockerServer, "docker_server", "d", "", "please enter docker server address")
	RootCmd.PersistentFlags().StringVarP(&externalIp, "externalIp", "e", "", "please enter server external ip address")
	RootCmd.PersistentFlags().StringVarP(&opSys, "opSys", "s", "", "please enter you op sys name : win、linux、mac")
	cobra.OnInitialize(initConfig)
}

// RootCmd represents the base command when called without any subcommands
var RootCmd = &cobra.Command{
	Use:   "miner",
	Short: "The miner command-line interface",
	Long:  ``,
	Run: func(cmd *cobra.Command, args []string) {
		if rewardAddr == "" {
			log.Error("Enter reward address is not nil")
			return
		}
		if opSys != "" {
			if !conf.GetConfig().SetOpSys(opSys) {
				log.Error("Enter op sys no support")
				return
			}
		}
		isSetDockerServer := conf.GetConfig().SetDockerServerUrl(dockerServer)
		if !isSetDockerServer {
			log.Error("Enter right docker server address:", dockerServer)
			return
		}
		log.Info("Enter docker server url:", dockerServer)
		isSetReward := conf.GetConfig().SetRewardAddress(rewardAddr)
		if !isSetReward {
			log.Error("Please set right reward address")
			return
		}
		conf.GetConfig().SetExternalIp(externalIp)
		log.Info("Enter reward address:", rewardAddr)
		log.InitLog(log.LogConfig{Path: "logs", Level: "debug", Save: 3})
		go nm.StartMonitor()
		beego.Run()
	},
}

func initConfig() {
	// 设置配置文件的名称（不包含扩展名)
	viper.SetConfigName("config")

	// 设置配置文件的类型
	viper.SetConfigType("json")

	// 设置配置文件所在的目录
	viper.AddConfigPath(".")

	viper.AutomaticEnv()

	// 读取配置文件
	if err := viper.ReadInConfig(); err != nil {
		fmt.Println("Error reading config file:", err)
		return
	}

	configFilePath := viper.ConfigFileUsed()
	if configFilePath == "" {
		// handle error
		log.Error("config file path is empty")
		panic("config file path is empty")
	}

	data, err := ioutil.ReadFile(configFilePath)
	if err != nil {
		// handle error
		log.Error("Read cfg file error:", err)
		panic("Read cfg file error")
	}
	err = json.Unmarshal(data, conf.GetConfig())
	if err != nil {
		// handle error
		log.Error("Json unmarshal cfg error:", err)
		panic("Json unmarshal cfg error")
	}
	conf.GetConfig().HeartRespTimeMillis = conf.GetConfig().HeartRespTimeSecond * 60 * 60 * 1000
	prvKey, err := utils.GetPrv()
	if err != nil {
		panic("get prv error or delete keystore after restart")
	}
	conf.GetConfig().SignPrivateKey = prvKey
	ecdsaPub := prvKey.PublicKey
	conf.GetConfig().SignPub = common.Bytes2Hex(crypto.FromECDSAPub(&ecdsaPub))
	log.Info("PublicKey", conf.GetConfig().SignPub)
	publicAddr := crypto.PubkeyToAddress(ecdsaPub)
	log.Info("publicAddr:", publicAddr)
	conf.GetConfig().SignPublicAddress = publicAddr
}

func Execute() {
	if err := RootCmd.Execute(); err != nil {
		log.Error("root cmd execute failed", err)
		os.Exit(-1)
	}
}
