package largeModel

import (
	"encoding/json"
	"example.com/m/conf"
	"example.com/m/log"
	"example.com/m/models"
	"example.com/m/operate"
	"fmt"
	"io"
	"net/http"
	"os"
	"strings"
	"time"
)

type ModelHandler struct {
	dockerOp       *operate.DockerOp
	client         *http.Client
	modelsFileName string
}

func NewModelHandler(dockerOp *operate.DockerOp) *ModelHandler {
	return &ModelHandler{
		dockerOp:       dockerOp,
		client:         &http.Client{},
		modelsFileName: "models.json",
	}
}

func (m *ModelHandler) MonitorModelInfo() {
	ticker := time.NewTicker(time.Second * 1)
	for {
		select {
		case <-ticker.C:
			modelResp, err := m.client.Get(conf.GetConfig().ApiUrl)
			if err != nil {
				log.Error("Error getting model info from client failed:", err)
				continue
			}
			bodyBytes, err := io.ReadAll(modelResp.Body)
			if err != nil {
				log.Error("Error reading model response failed:", err)
				continue
			}
			resp := &models.Resp{}
			err = json.Unmarshal(bodyBytes, resp)
			if err != nil {
				log.Error("Unmarshal model response failed:", err)
				continue
			}
			if resp.Code != http.StatusOK {
				log.Error("Response code :", resp.Code)
				continue
			}
			if resp.Data == nil || len(resp.Data) == 0 {
				log.Warn("Response data is empty")
				continue
			}
			modelInfosResp := resp.Data
			imageNameMap, err := m.dockerOp.PsImageNameMap()
			if err != nil {
				log.Error("Docker op ps images failed:", err)
				continue
			}
			reportTaskIds := make([]uint64, 0)
			for _, modelInfo := range modelInfosResp {
				if modelInfo.ImageName == "" {
					continue
				}
				modelInfo.ImageName = fmt.Sprintf("%s-%s", modelInfo.ImageName, conf.GetConfig().OpSys)
				split := strings.Split(modelInfo.ImageName, ":")
				if len(split) != 2 {
					continue
				}
				if !imageNameMap[modelInfo.ImageName] {
					// todo: 判断机器资源是否够用
					isPull := isResourceEnough(modelInfo)
					// todo： 如果够用
					if isPull && modelInfo.PublishStatus == models.ModelPublishStatusYes {
						log.WithField("model image name", modelInfo.ImageName).Info("pulling image")
						go m.dockerOp.PullImage(modelInfo)
					}
				} else {
					log.WithField("name", modelInfo.ImageName).Info("The image name is already")
					reportTaskIds = append(reportTaskIds, modelInfo.TaskId)
				}
				m.dockerOp.SignApi[modelInfo.ImageName] = modelInfo.SignUrl
			}
			m.dockerOp.ModelsInfo = modelInfosResp
			m.dockerOp.ReportTaskIds = reportTaskIds
			err = os.WriteFile(m.modelsFileName, bodyBytes, 0644)
			if err != nil {
				log.WithError(err).Error("Error writing models.json")
			}
			ticker = time.NewTicker(time.Minute * 10)
		}
	}
}

func (m *ModelHandler) ReadModels() ([]*models.ModelInfo, error) {
	bodyBytes, err := os.ReadFile(m.modelsFileName)
	if err != nil {
		log.WithError(err).WithField("fileName", m.modelsFileName).Error("Error reading")
		return nil, err
	}
	resp := &models.Resp{}
	err = json.Unmarshal(bodyBytes, resp)
	if err != nil {
		log.WithField("fileName", m.modelsFileName).Error("Unmarshal model response failed:", err)
		return nil, err
	}
	if resp.Code != http.StatusOK {
		log.WithField("fileName", m.modelsFileName).Error("Response code :", resp.Code)
		return nil, err
	}
	if resp.Data == nil || len(resp.Data) == 0 {
		log.WithField("fileName", m.modelsFileName).Warn("Response data is empty")
		return nil, err
	}
	return resp.Data, nil
}

func isResourceEnough(modelInfo *models.ModelInfo) bool {
	return true
}
