package nm

import (
	"bytes"
	"encoding/json"
	"example.com/m/conf"
	"example.com/m/log"
	"example.com/m/models"
	"example.com/m/operate"
	"example.com/m/utils"
	"fmt"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/crypto"
	"github.com/golang/groupcache/lru"
	baseV1 "github.com/odysseus/odysseus-protocol/gen/proto/go/base/v1"
	nodeManagerV1 "github.com/odysseus/odysseus-protocol/gen/proto/go/nodemanager/v1"
	"io"
	"math/rand"
	"mime/multipart"
	"net/http"
	"net/url"
	"strconv"
	"strings"
	"sync"
	"time"
)

type TaskHandler struct {
	Wg                 *sync.WaitGroup
	Mutex              *sync.Mutex
	LruCache           *lru.Cache
	DockerOp           *operate.DockerOp
	CmdOp              *operate.Command
	TaskMsg            chan *nodeManagerV1.PushTaskMessage
	HttpClient         *http.Client
	IsExecAiTask       bool
	IsExecStandardTask bool
}

var oldTaskImageName string

func NewTaskWorker(op *operate.DockerOp) *TaskHandler {
	return &TaskHandler{
		Wg:           &sync.WaitGroup{},
		Mutex:        &sync.Mutex{},
		LruCache:     lru.New(100),
		DockerOp:     op,
		TaskMsg:      make(chan *nodeManagerV1.PushTaskMessage, 0),
		HttpClient:   &http.Client{},
		IsExecAiTask: false,
	}
}

func (t *TaskHandler) HandlerTask(runCount int) {
	for i := 0; i < runCount; i++ {
		go func(t *TaskHandler) {
			for {
				select {
				case taskMsg := <-t.TaskMsg:
					{
						switch taskMsg.TaskKind {
						case baseV1.TaskKind_SystemTask:
							{
								//command := operate.GetSystemCommand(taskMsg.TaskCmd, taskMsg.TaskParam, taskMsg.TaskId+".sh")
								t.SystemTaskHandler(taskMsg)
							}
						case baseV1.TaskKind_ComputeTask:
							{
								t.ComputeTaskHandler(taskMsg)
							}
						case baseV1.TaskKind_CustomTask:
							{
								t.CustomTaskHandler(taskMsg)
							}
						case baseV1.TaskKind_StandardTask:
							{
								t.ComputeTaskHandler(taskMsg)
							}
						}
					}
				}
			}
		}(t)
	}
}

func (t *TaskHandler) SystemTaskHandler(taskMsg *nodeManagerV1.PushTaskMessage) {
	defer t.Wg.Done()

	log.Info("received systemTask--------------------------------")
}

func (t *TaskHandler) ComputeTaskHandler(taskMsg *nodeManagerV1.PushTaskMessage) {
	defer t.Wg.Done()
	taskExecResult := &models.TaskResult{
		TaskHttpStatusCode: 200,
		TaskRespBody:       nil,
		TaskHttpHeaders:    nil,
		TaskIsSuccess:      false,
		TaskExecTime:       0,
		TaskExecError:      "",
	}
	taskCmd := &models.TaskCmd{}
	err := json.Unmarshal(bytes.NewBufferString(taskMsg.TaskCmd).Bytes(), taskCmd)
	if err != nil {
		log.Errorf("failed to unmarshal task cmd: %s", err.Error())
		taskExecResult.TaskExecError = fmt.Sprintf("%s,%s", "failed to unmarshal task cmd: %s", err.Error())
		return
	}
	log.Info("received task cmd :", taskCmd)

	if taskMsg.TaskKind == baseV1.TaskKind_ComputeTask {
		t.IsExecAiTask = true
		if t.IsExecStandardTask {
			//todo: 停止标准任务容器
			//containers := t.DockerOp.ListContainer()
			//for _, container := range containers {
			//	if container.Image == taskCmd.ImageName && container.State == "running" {
			//		t.DockerOp.StopContainer(container.ID)
			//	}
			//}
			t.IsExecStandardTask = false
		}
	} else if taskMsg.TaskKind == baseV1.TaskKind_StandardTask {
		t.IsExecStandardTask = true
	}
	if oldTaskImageName != "" && oldTaskImageName != taskCmd.ImageName {
		//todo: 停止标准任务容器
		containers := t.DockerOp.ListContainer()
		for _, container := range containers {
			split := strings.Split(container.Image, ":")
			if len(split) == 1 {
				container.Image = fmt.Sprintf("%s:%s", container.Image, "latest")
			}
			if container.Image == taskCmd.ImageName && container.State == "running" {
				t.DockerOp.StopContainer(container.ID)
			}
		}
		oldTaskImageName = taskCmd.ImageName
	}
	images, err := t.DockerOp.PsImages()
	if err != nil {
		log.Error("Ps images failed:", err)
		taskExecResult.TaskExecError = fmt.Sprintf("%s,%s", "Ps images failed:", err.Error())
		return
	}
	imageId := ""
	isFound := false
	for _, image := range images {
		if isFound {
			break
		}
		for _, tag := range image.RepoTags {
			if tag == taskCmd.ImageName {
				imageId = image.ID
				isFound = true
				log.Info("The image found success:", image.ID)
				break
			}
		}
	}
	if !isFound {
		log.Error("The image is not found:", taskCmd.ImageName)
		taskExecResult.TaskExecError = fmt.Sprintf("%s,%s", "The image is not found:", taskCmd.ImageName)
		return
	}
	running, internalIp, internalPort := t.foundImageIsRunning(imageId)
	if !running {
		var externalPort int64
		for {
			// 设置种子以确保每次运行时生成不同的随机数序列
			rand.Seed(time.Now().UnixNano())
			// 生成一个介于 0 和 100 之间的随机整数
			externalPort = rand.Int63n(10001) + 10000
			log.Info("DockerOp UsedExternalPort :", t.DockerOp.UsedExternalPort[externalPort])
			if t.DockerOp.UsedExternalPort[externalPort] {
				continue
			}
			break
		}
		taskCmd.DockerCmd.HostIp = "0.0.0.0"
		taskCmd.DockerCmd.HostPort = strconv.FormatInt(externalPort, 10)
		containerId, err := t.DockerOp.CreateAndStartContainer(taskCmd.ImageName, taskCmd.DockerCmd)
		if err != nil {
			log.Errorf("Create and start container failed: %s", err.Error())
			taskExecResult.TaskExecError = fmt.Sprintf("%s,%s", "Create and start container failed: %s", err.Error())
			return
		}
		log.Infof("Started container with ID %s", containerId)
		time.Sleep(time.Second * 40)
		running, internalIp, internalPort = t.foundImageIsRunning(imageId)
		if running {
			taskCmd.ApiUrl = fmt.Sprintf("http://%s:%d%s", internalIp, internalPort, taskCmd.ApiUrl)
			log.Info("Container ports:", internalPort)
			log.WithField("ApiUrl", taskCmd.ApiUrl).Info("The image is not  running")
		}
	} else {
		taskCmd.ApiUrl = fmt.Sprintf("http://%s:%d%s", internalIp, internalPort, taskCmd.ApiUrl)
		log.Info("Container ports:", internalPort)
		log.WithField("ApiUrl", taskCmd.ApiUrl).Info("The image is running")
	}
	startBeforeTaskTime := time.Now()
	taskParam := &models.TaskParam{}
	err = json.Unmarshal(taskMsg.TaskParam, taskParam)
	if err != nil {
		log.WithField("err", err).Error("Error unmarshalling task parameter")
		taskExecResult.TaskExecError = fmt.Sprintf("%s,%s", "Error unmarshalling task parameter", err.Error())
		return
	}
	reqContainerBody := bytes.NewReader(taskParam.Body)
	if len(taskParam.Queries) > 0 {
		queryString := utils.MatchContainerQueryString(taskParam.Queries)
		taskCmd.ApiUrl = fmt.Sprintf("%s?%s", taskCmd.ApiUrl, queryString)
	}
	request, err := http.NewRequest("POST", taskCmd.ApiUrl, reqContainerBody)
	request.Header.Set("Content-Type", "application/json")
	for key, value := range taskParam.Headers {
		if key == models.Prefer {
			if value[0] == models.Async {
				request.Header.Set(models.Prefer, models.Async)
				m := &models.ContainerRequest{}
				err := json.Unmarshal(taskParam.Body, m)
				if err != nil {
					log.WithError(err).Error("json unmarshal task body failed")
					taskExecResult.TaskExecError = fmt.Sprintf("%s,%s", "Json unmarshal task body failed", err.Error())
					return
				}
				if m.WebHook == "" {
					log.Error("Request webhook is nil")
					taskExecResult.TaskExecError = fmt.Sprintf("%s", "Request webhook is nil")
					return
				} else {
					_, err := url.Parse(m.WebHook)
					if err != nil {
						log.WithError(err).Error("web hook url parse failed")
						taskExecResult.TaskExecError = fmt.Sprintf("%s,%s", "web hook url parse failed", err.Error())
						return
					}
				}
				break
			}
		}
	}
	if err != nil {
		log.WithField("error:", err).Error("New container request failed")
		taskExecResult.TaskExecError = fmt.Sprintf("%s,%s", "Http client post request container failed", err.Error())
		return
	}
	post, err := t.HttpClient.Do(request)
	if err != nil {
		log.WithField("error:", err).Error("Http client post request container failed")
		taskExecResult.TaskExecError = fmt.Sprintf("%s,%s", "Http client post request container failed", err.Error())
		return
	}
	endAfterTaskTime := time.Since(startBeforeTaskTime)
	log.WithField("time", endAfterTaskTime.Seconds()).WithField("taskId", taskMsg.TaskId).Info("Exec task end (second is units) :")
	log.WithField("StatusCode", post.StatusCode).WithField("taskId", taskMsg.TaskId).Info("Exec task result")
	if post.StatusCode == http.StatusOK {
		taskExecResult.TaskHttpStatusCode = http.StatusOK
		readBody, err := io.ReadAll(post.Body)
		if err != nil {
			taskExecResult.TaskExecError = fmt.Sprintf("%s,%s,Container Http Code:%d", "Read container body failed", err.Error(), post.StatusCode)
			log.Error("Read container body failed", err)
			return
		}
		isUseFileCache := true
		if taskMsg.TaskKind != baseV1.TaskKind_StandardTask {
			for key, value := range taskParam.Headers {
				if key == models.UseFileCache {
					if value[0] == "0" {
						isUseFileCache = false
						break
					}
				}
			}
			log.WithField("isUseFileCache", isUseFileCache).Info("is use file cache")
			if readBody != nil {
				data := parseData(readBody)
				if data != nil {
					switch v := data.(type) {
					case [][]string:
						{
							res := data.([][]string)
							log.Info("data is [][]string type")
							apiRes := make([][]string, 0)
							for _, innerSlice := range res {
								apiResOneArr := make([]string, 0)
								for _, respStr := range innerSlice {
									if !isUseFileCache {
										apiResOneArr = append(apiResOneArr, respStr)
										continue
									}
									ossUri, err := t.getFileCache(respStr, taskMsg, taskParam, taskCmd)
									if err != nil || ossUri == "" {
										apiResOneArr = append(apiResOneArr, respStr)
										continue
									}
									if ossUri != "" && len(res) == 1 && len(innerSlice) == 1 {
										taskExecResult.TaskHttpStatusCode = models.RedirectCode
										post.Header.Set("Location", ossUri)
										break
									} else {
										apiResOneArr = append(apiResOneArr, ossUri)
									}
								}
								apiRes = append(apiRes, apiResOneArr)
							}
							if len(apiRes) > 1 || len(apiRes[0]) > 1 {
								apiResBody := utils.EncodeJsonEscapeHTML(apiRes)
								taskExecResult.TaskRespBody = apiResBody
							}
						}
					case []string:
						{
							res := data.([]string)
							log.Info("data is []string type")
							apiRes := make([]string, 0)
							for _, respStr := range res {
								if !isUseFileCache {
									apiRes = append(apiRes, respStr)
									continue
								}
								ossUri, err := t.getFileCache(respStr, taskMsg, taskParam, taskCmd)
								if err != nil || ossUri == "" {
									apiRes = append(apiRes, respStr)
									continue
								}
								if ossUri != "" && len(res) == 1 {
									taskExecResult.TaskHttpStatusCode = models.RedirectCode
									post.Header.Set("Location", ossUri)
									break
								} else {
									apiRes = append(apiRes, ossUri)
								}
							}
							if len(apiRes) > 1 {
								apiResBody := utils.EncodeJsonEscapeHTML(apiRes)
								taskExecResult.TaskRespBody = apiResBody
							}
						}
					case string:
						{
							resStr := data.(string)
							log.Info("data is string type")
							resArr := []string{resStr}
							apiRes := make([]string, 0)
							for _, respStr := range resArr {
								if !isUseFileCache {
									apiRes = append(apiRes, respStr)
									continue
								}
								ossUri, err := t.getFileCache(respStr, taskMsg, taskParam, taskCmd)
								if err != nil || ossUri == "" {
									apiRes = append(apiRes, respStr)
									continue
								}
								if ossUri != "" && len(resArr) == 1 {
									taskExecResult.TaskHttpStatusCode = models.RedirectCode
									post.Header.Set("Location", ossUri)
									break
								} else {
									apiRes = append(apiRes, ossUri)
								}
							}
							if len(apiRes) > 1 {
								apiResBody := utils.EncodeJsonEscapeHTML(apiRes)
								taskExecResult.TaskRespBody = apiResBody
							}
						}
					default:
						log.Error("data is unknown type", v)
						taskExecResult.TaskExecError = "Container resp data is unknown type"
						apiRes := make([]string, 0)
						apiResBody := utils.EncodeJsonEscapeHTML(apiRes)
						taskExecResult.TaskRespBody = apiResBody
					}
				}
			}
		}
		headers, err := json.Marshal(post.Header)
		if err != nil {
			log.WithError(err).Error("JSON marshal container header failed")
			taskExecResult.TaskExecError = fmt.Sprintf("%s,%s", "JSON marshal container header failed", err.Error())
			return
		}
		log.WithField("headers", post.Header).Info("return task http headers")
		taskExecResult.TaskHttpHeaders = headers
		taskExecResult.TaskIsSuccess = true
		taskExecResult.TaskExecTime = endAfterTaskTime.Microseconds()
	} else {
		taskExecResult.TaskHttpStatusCode = int32(post.StatusCode)
		if post.Body != nil {
			all, err := io.ReadAll(post.Body)
			if err != nil {
				log.Error("JSON read error: ", err)
				return
			}
			taskExecResult.TaskExecError = fmt.Sprintf("%s,Container Http Code:%d,body:%s", "Read container body failed", post.StatusCode, string(all))
		} else {
			taskExecResult.TaskExecError = fmt.Sprintf("%s,Container Http Code:%d,body:%s", "Read container body failed", post.StatusCode, "")
		}
		apiRes := make([]string, 0)
		apiResBody := utils.EncodeJsonEscapeHTML(apiRes)
		taskExecResult.TaskRespBody = apiResBody
		log.WithField("error", post.Body).WithField("taskId", taskMsg.TaskId).Error("Exec task result is failed")
	}
	if taskMsg.TaskKind == baseV1.TaskKind_ComputeTask {
		t.IsExecAiTask = false
	} else if taskMsg.TaskKind == baseV1.TaskKind_StandardTask {
		t.IsExecStandardTask = false
	}
	t.Mutex.Lock()
	t.LruCache.Add(taskMsg.TaskId, taskExecResult)
	t.Mutex.Unlock()
	//log.WithField("result", taskExecResult).Info("lru cache storage task result")
	log.Info("received computeTask--------------------------------")
}

func (t *TaskHandler) CustomTaskHandler(taskMsg *nodeManagerV1.PushTaskMessage) {
	defer t.Wg.Done()
	_, err := t.DockerOp.PsImages()
	if err != nil {
		log.Error("custom task handler docker op ps images failed: ", err)
		return
	}
	log.Info("received customTask--------------------------------")
}

func (t *TaskHandler) GetMinerSign(msg *nodeManagerV1.PushTaskMessage, taskResult []byte) ([]byte, []byte, []byte) {
	reqHash := crypto.Keccak256Hash(msg.TaskParam)
	respHash := crypto.Keccak256Hash(taskResult)
	signHash := crypto.Keccak256Hash(bytes.NewBufferString(msg.TaskId).Bytes(), reqHash.Bytes(), respHash.Bytes())
	log.WithField("hash", signHash.String()).Info("Miner sign result")
	sign, err := crypto.Sign(signHash.Bytes(), conf.GetConfig().SignPrivateKey)
	if err != nil {
		log.Error("custom task handler")
	}
	log.Info("Miner sign:", common.Bytes2Hex(sign))
	return reqHash.Bytes(), respHash.Bytes(), sign
}

func (t *TaskHandler) foundImageIsRunning(imageId string) (bool, string, uint16) {
	containers := t.DockerOp.ListContainer()
	for _, container := range containers {
		if container.ImageID == imageId && container.State == "running" {
			networks := container.NetworkSettings.Networks
			ip := ""
			for _, endPoint := range networks {
				ip = endPoint.IPAddress
				log.Warn("Container network ip:", ip)
			}
			return true, ip, container.Ports[0].PrivatePort
		}
	}
	return false, "", 0
}

func (t *TaskHandler) uploadOSS(taskId string, queries string, decodedImage []byte, suffix string) (string, error) {
	var requestBody bytes.Buffer
	writer := multipart.NewWriter(&requestBody)
	// 创建文件表单字段
	fileField, err := writer.CreateFormFile("file", fmt.Sprintf("%s.%s", taskId, suffix))
	if err != nil {
		log.WithError(err).Error("Error creating form file")
		return "", err
	}
	_, err = io.Copy(fileField, bytes.NewReader(decodedImage))
	//_, err = io.Copy(fileField, strings.NewReader(base64Image))
	if err != nil {
		log.WithError(err).Error("Error copying file contents")
		return "", err
	}
	// 关闭 multipart writer
	err = writer.Close()
	if err != nil {
		log.WithError(err).Error("Error closing writer")
		return "", err
	}
	ossUrl := fmt.Sprintf("%s?%s", conf.GetConfig().OssUrl, queries)
	request, err := http.NewRequest("POST", ossUrl, &requestBody)
	if err != nil {
		return "", err
	}
	request.Header.Set("Content-Type", writer.FormDataContentType())
	response, err := t.HttpClient.Do(request)
	if err != nil {
		log.WithError(err).Error("Error request oss failed")
		return "", err
	}
	ossRespBody, err := io.ReadAll(response.Body)
	if err != nil {
		log.WithError(err).Error("Error read oss resp body failed")
		return "", err
	}
	fileCacheRes := &models.FileCacheResult{}
	err = json.Unmarshal(ossRespBody, fileCacheRes)
	if err != nil {
		log.WithError(err).Error("Json unmarshal file cache result failed")
		return "", err
	}
	log.WithField("code", fileCacheRes.Code).WithField("msg", fileCacheRes.Msg).WithField("data", fileCacheRes.Data).Info("file cache result")
	if fileCacheRes.Code == http.StatusOK && fileCacheRes.Data != "" {
		_, err := url.Parse(fileCacheRes.Data)
		if err != nil {
			log.WithError(err).Error("url parse file cache data error")
			return "", err
		}
		return fileCacheRes.Data, nil
	}
	return "", err
}

func (t *TaskHandler) getFileCache(respStr string, taskMsg *nodeManagerV1.PushTaskMessage, taskParam *models.TaskParam, taskCmd *models.TaskCmd) (string, error) {
	isBase64, decodeByte, respFormat, suffix := utils.IsBase64ImageStr(respStr)
	log.WithField("isBase64", isBase64).Info("resp str info")
	if isBase64 {
		log.WithField("taskId", taskMsg.TaskId).WithField("format", respFormat).WithField("suffix", suffix).Info("Parse container resp")
		queryString := utils.MatchFileCacheQueryString(taskParam.Headers, taskCmd.ImageName, t.DockerOp.ModelsInfo, respFormat)
		ossUri, err := t.uploadOSS(taskMsg.TaskId, queryString, decodeByte, suffix)
		if err != nil || ossUri == "" {
			log.WithError(err).Error("upload image into file cache failed")
			return "", err
		}
		log.WithField("uri", ossUri).Info("upload image OSS  successful")
		return ossUri, nil
	}
	return "", nil
}

func parseData(readBody []byte) interface{} {
	var m map[string]json.RawMessage
	if err := json.Unmarshal(readBody, &m); err != nil {
		log.WithError(err).Error("Parse json raw message failed")
		return bytes.NewBuffer(readBody).String()
	}
	var outputTwoArray [][]string
	if err := json.Unmarshal(m["output"], &outputTwoArray); err != nil {
		log.WithField("err", err).Warn("parse two array output filed failed:")
		var outputOneArray []string
		if err := json.Unmarshal(m["output"], &outputOneArray); err != nil {
			log.WithField("err", err).Warn("parse one array output filed failed:")
			var outputString string
			if err := json.Unmarshal(m["output"], &outputString); err != nil {
				return nil
			} else {
				return outputString
			}
		} else {
			return outputOneArray
		}
	} else {
		return outputTwoArray
	}
}
