package main

import (
	"example.com/m/conf"
	"example.com/m/log"
	"example.com/m/models"
	"example.com/m/nm"
	"example.com/m/utils"
	"fmt"
	"github.com/astaxie/beego"
	"github.com/fsnotify/fsnotify"
	"github.com/spf13/cobra"
	"github.com/spf13/viper"
	"os"
	"strings"
)

var (
	rewardAddr, externalIp, opSys string
	debug                         bool
)

func init() {
	RootCmd.PersistentFlags().StringVarP(&rewardAddr, "reward", "r", "0x40EC4256fcBCA69CdbAc942594caeC79FBE10494", "please enter a reward address")
	RootCmd.PersistentFlags().StringVarP(&externalIp, "externalIp", "e", "192.168.1.102", "please enter server external ip address")
	RootCmd.PersistentFlags().StringVarP(&opSys, "opSys", "s", "", "please enter you op sys name : win、linux")
	RootCmd.PersistentFlags().BoolVarP(&debug, "debug", "d", false, "set log level debug")
	cobra.OnInitialize(initConfig)
}

// RootCmd represents the base command when called without any subcommands
var RootCmd = &cobra.Command{
	Use:   "miner",
	Short: "The miner command-line interface",
	Long:  ``,
	Run: func(cmd *cobra.Command, args []string) {
		if opSys != "" {
			if !conf.GetConfig().SetOpSys(opSys) {
				log.Error("Enter op sys no support")
				return
			}
		}
		conf.GetConfig().SetExternalIp(externalIp)
		log.Info("Enter reward address:", rewardAddr)
		if debug {
			log.InitLog(log.LogConfig{Path: "logs", Level: "debug", Save: 3})
		} else {
			log.InitLog(log.LogConfig{Path: "logs", Level: "info", Save: 3})
		}
		if rewardAddr != "" {
			isSetBenefit := conf.GetConfig().SetBenefitAddress(rewardAddr)
			if !isSetBenefit {
				log.Error("Please set right reward address")
				return
			}
			fileBenefitAcc, _ := utils.ReadBenefitFile()
			if fileBenefitAcc != nil && len(fileBenefitAcc) > 0 {
				nm.HistoryBenefitAcc = fileBenefitAcc
			}
			isExist := false
			for _, acc := range nm.HistoryBenefitAcc {
				if strings.ToLower(acc.Address) == strings.ToLower(rewardAddr) {
					isExist = true
				}
			}
			if !isExist {
				nm.HistoryBenefitAcc = append(nm.HistoryBenefitAcc, &models.BenefitAddressStruct{Address: rewardAddr, IsDel: false})
				err := utils.WriteBenefitFile(nm.HistoryBenefitAcc)
				if err != nil {
					log.Error("WriteBenefitFile failed with error:", err)
					return
				}
			}
			go nm.StartMonitor()
		}
		beego.Run()
	},
}

func initConfig() {
	// 设置配置文件的名称（不包含扩展名)
	viper.SetConfigName("config")

	// 设置配置文件的类型
	viper.SetConfigType("json")

	// 设置配置文件所在的目录
	viper.AddConfigPath(".")

	viper.AutomaticEnv()

	viper.WatchConfig()

	viper.OnConfigChange(func(e fsnotify.Event) {
		// 配置文件发生变更之后会调用的回调函数
		log.Warn("The configuration file has been modified...........")
		err := viper.Unmarshal(conf.GetConfig())
		if err != nil {
			log.WithError(err).Error("Viper unmarshal cfg error:")
			panic(fmt.Errorf("Viper unmarshal conf failed, err:%s \n", err))
		}
		conf.GetConfig().UpdateFiledInfo()
		log.Info("Config file changed success:", e.Name)
	})

	// 读取配置文件
	if err := viper.ReadInConfig(); err != nil {
		panic(fmt.Errorf("Error reading config file: %s ", err.Error()))
	}

	err := viper.Unmarshal(conf.GetConfig())
	if err != nil {
		// handle error
		log.Error("Viper unmarshal cfg error:", err)
		panic("Viper unmarshal cfg error")
	}
	conf.GetConfig().UpdateFiledInfo()
}

func Execute() {
	if err := RootCmd.Execute(); err != nil {
		log.Error("root cmd execute failed", err)
		os.Exit(-1)
	}
}
