package utils

import (
	"bytes"
	"crypto/aes"
	"crypto/cipher"
	"crypto/ecdsa"
	"crypto/elliptic"
	"crypto/rand"
	"encoding/base64"
	"encoding/json"
	"example.com/m/db"
	"example.com/m/log"
	"example.com/m/models"
	//"example.com/m/nm"
	"fmt"
	"github.com/ethereum/go-ethereum/accounts/keystore"
	"github.com/ethereum/go-ethereum/crypto"
	"io"
	"math/big"
	"net/http"
	"net/url"
	"os"
	"strings"
)

const KeypadPwd = "keystore"

func GenerateRandomNumber(privateKey *ecdsa.PrivateKey, length int64) *big.Int {
	// 生成一个随机数
	randNum, err := rand.Int(rand.Reader, elliptic.P256().Params().N)
	if err != nil {
		log.Error("Error generating random number:", err)
	}
	randNum.Mod(randNum, big.NewInt(length))
	log.Info("Generating random number:", randNum)
	return randNum
}

func GetPrv() (*ecdsa.PrivateKey, error) {
	if _, err := os.Stat(KeypadPwd); os.IsNotExist(err) {
		//log.Info("Keystore not found. Generating a new one...")
		// 生成私钥
		privateKey, err := generatePrivateKey()
		if err != nil {
			log.Error("Error generating private key:", err)
			return nil, err
		}
		// 保存私钥到 keystore 文件
		err = savePrivateKey(privateKey)
		if err != nil {
			log.Error("Error saving private key:", err)
			return nil, err
		}
		//log.Info("Keystore generated successfully.")
		return privateKey, nil
	} else {
		//log.Info("Keystore found. Reading private key...")
		// 读取私钥
		privateKey, err := readPrivateKey()
		if err != nil || privateKey == nil {
			log.Error("Error reading private key:", err)
			return nil, err
		}
		//log.Info("Private key read successfully:", privateKey)
		return privateKey, nil
	}
}

func generatePrivateKey() (*ecdsa.PrivateKey, error) {
	return crypto.GenerateKey()
}

func savePrivateKey(privateKey *ecdsa.PrivateKey) error {
	ks := keystore.NewKeyStore(KeypadPwd, keystore.StandardScryptN, keystore.StandardScryptP)
	account, err := ks.ImportECDSA(privateKey, KeypadPwd)
	if err != nil {
		log.Fatal(err)
	}
	fmt.Println(account.Address.Hex())
	return nil
}

func readPrivateKey() (*ecdsa.PrivateKey, error) {
	file, err := os.ReadDir(KeypadPwd)
	if err != nil {
		return nil, err
	}
	for _, info := range file {
		keystoreFile := fmt.Sprintf("%s%s%s", KeypadPwd, "/", info.Name())
		jsonBytes, err := os.ReadFile(keystoreFile)
		if err != nil {
			log.Error("import ecdsa keystore error: ", err)
			continue
		}
		key, err := keystore.DecryptKey(jsonBytes, KeypadPwd)
		if err != nil {
			log.Error("keystore decrypt key failed:", err)
			continue
		}
		return key.PrivateKey, nil
	}
	return nil, nil
}

// IsBase64ImageStr 检查字符串是否是 Base64 编码的图像数据
func IsBase64ImageStr(imageStr string) (bool, []byte, string, string) {
	// 移除可能的前缀（如 "data:image/png;base64,"）
	imageStrArr := strings.Split(imageStr, ",")
	if len(imageStrArr) < 2 {
		return false, nil, "", ""
	}
	base64CodePrefix := imageStrArr[0]
	base64Code := imageStrArr[1]
	decodeBytes, err := base64.StdEncoding.DecodeString(base64Code)
	if err != nil {
		log.WithError(err).Error("base64 decode string failed")
		return false, nil, "", ""
	}
	dataSuffix := strings.Split(base64CodePrefix, ";")
	if len(dataSuffix) < 2 {
		return false, nil, "", ""
	}
	log.Info("dataSuffix:", dataSuffix)
	formatStrArr := strings.Split(dataSuffix[0], ":")
	if len(formatStrArr) < 2 {
		return false, nil, "", ""
	}
	formatStr := formatStrArr[1]
	log.Info("formatStr:", formatStr)
	suffixArr := strings.Split(formatStr, "/")
	if len(suffixArr) < 2 {
		return false, nil, "", ""
	}
	suffix := suffixArr[1]
	return true, decodeBytes, formatStr, suffix
}

func MatchFileCacheQueryString(params map[string][]string, taskImageName string, contentType string) string {
	values := url.Values{}
	isExistFileExpires := false
	for key, value := range params {
		if key == models.ResultFileExpiresDB {
			values.Add(key, value[0])
			isExistFileExpires = true
			break
		}
	}
	if !isExistFileExpires {
		modelInfo, _ := db.GetModel(taskImageName)
		if modelInfo != nil && modelInfo.FileExpiresTime != "" {
			values.Add(models.ResultFileExpiresDB, modelInfo.FileExpiresTime)
		} else {
			values.Add(models.ResultFileExpiresDB, "600")
		}
	}
	values.Add(models.ContentType, contentType)
	return values.Encode()
}

func MatchContainerQueryString(params map[string]string) string {
	values := url.Values{}
	for key, value := range params {
		values.Add(key, value)
	}
	return values.Encode()
}

func EncodeJsonEscapeHTML(apiRes any) []byte {
	apiResBody := bytes.NewBuffer([]byte{})
	encoder := json.NewEncoder(apiResBody)
	encoder.SetEscapeHTML(false)
	err := encoder.Encode(apiRes)
	if err != nil {
		log.WithError(err).Error("encoder Encode")
		return apiResBody.Bytes()
	}
	//log.WithField("apiResBody", string(apiResBody.Bytes())).Info("model resp")
	return apiResBody.Bytes()
}

func BuildParams(params ...interface{}) []interface{} {
	res := make([]interface{}, len(params))
	for i, param := range params {
		res[i] = param
	}
	return res
}

func CloneRequest(r *http.Request) *http.Request {
	copiedRequest := &http.Request{
		Method:           r.Method,
		URL:              r.URL,
		Proto:            r.Proto,
		ProtoMajor:       r.ProtoMajor,
		ProtoMinor:       r.ProtoMinor,
		Header:           make(http.Header),
		Body:             r.Body,
		ContentLength:    r.ContentLength,
		TransferEncoding: append([]string(nil), r.TransferEncoding...),
		Close:            r.Close,
		Host:             r.Host,
		RemoteAddr:       r.RemoteAddr,
		RequestURI:       r.RequestURI,
		TLS:              r.TLS,
	}
	for k, v := range r.Header {
		copiedRequest.Header[k] = v
	}
	return copiedRequest
}

func DeleteNm(nodeArr []*models.NodeManagerClient, target int) []*models.NodeManagerClient {
	nodeArr[target] = nodeArr[len(nodeArr)-1]
	return nodeArr[:len(nodeArr)-1]
}

func WriteBenefitFile(historyBenefitAcc []*models.BenefitAddressStruct) error {
	log.Info("WritingBenefitFile........")
	key := []byte(models.EncryptionKey)
	serializedData, err := json.Marshal(historyBenefitAcc)
	if err != nil {
		log.Error("")
		return fmt.Errorf("json marshal HistoryBenefitAcc")
	}
	err = encryptAndWriteToFile(key, serializedData, models.HistoryBenefitAddressDirectory)
	if err != nil {
		return err
	}
	return nil
}

func ReadBenefitFile() ([]*models.BenefitAddressStruct, error) {
	log.Info("ReadBenefitFile........")
	key := []byte(models.EncryptionKey)
	readRes, err := readAndDecryptFile(key, models.HistoryBenefitAddressDirectory)
	if err != nil {
		return nil, err
	}
	res := make([]*models.BenefitAddressStruct, 0)
	err = json.Unmarshal(readRes, &res)
	if err != nil {
		return nil, err
	}
	return res, nil
}

func encryptAndWriteToFile(key []byte, serializedData []byte, filename string) error {
	// 创建 AES 加密器
	block, err := aes.NewCipher(key)
	if err != nil {
		return fmt.Errorf("new cipher error:%v", err)
	}

	// 使用 AES-GCM 模式加密
	gcm, err := cipher.NewGCM(block)
	if err != nil {
		return fmt.Errorf("new GCM error:%v", err)
	}

	// 生成随机 nonce
	nonce := make([]byte, gcm.NonceSize())
	if _, err = io.ReadFull(rand.Reader, nonce); err != nil {
		return fmt.Errorf("failed to read nonce:%v", err)
	}

	// 加密数据
	encryptedData := gcm.Seal(nonce, nonce, serializedData, nil)

	// 写入加密后的数据到文件
	if err := os.WriteFile(filename, encryptedData, 0644); err != nil {
		return fmt.Errorf("write file failed %v", err)
	}

	return nil
}

func readAndDecryptFile(key []byte, filename string) ([]byte, error) {
	// 读取加密文件内容
	encryptedData, err := os.ReadFile(filename)
	if err != nil {
		return nil, err
	}

	// 创建 AES 解密器
	block, err := aes.NewCipher(key)
	if err != nil {
		return nil, err
	}

	// 使用 AES-GCM 模式解密
	gcm, err := cipher.NewGCM(block)
	if err != nil {
		return nil, err
	}

	// 从加密文件中提取 nonce
	nonceSize := gcm.NonceSize()
	if len(encryptedData) < nonceSize {
		return nil, err
	}
	nonce, encryptedData := encryptedData[:nonceSize], encryptedData[nonceSize:]

	// 解密数据
	decryptedData, err := gcm.Open(nil, nonce, encryptedData, nil)
	if err != nil {
		return nil, err
	}

	return decryptedData, nil
}

func GetApiHardwareInfo(url string) *models.HardwareInfoRep {
	resp, err := http.Get(url)
	if err != nil {
		log.Error("Error creating request")
		return nil
	}
	res := &models.HardwareInfoRep{}
	body, err := io.ReadAll(resp.Body)
	if err != nil {
		log.Error("io.ReadAll failed")
		return nil
	}
	err = json.Unmarshal(body, res)
	if err != nil {
		log.Error("json.Unmarshal failed")
		return nil
	}
	return res
}
