package operate

import (
	"bytes"
	"context"
	"encoding/json"
	"example.com/m/log"
	"example.com/m/models"
	"fmt"
	"github.com/docker/docker/api/types"
	"github.com/docker/docker/api/types/container"
	"github.com/docker/docker/client"
	"github.com/docker/go-connections/nat"
	"github.com/ethereum/go-ethereum/common"
	nodemanagerv1 "github.com/odysseus/odysseus-protocol/gen/proto/go/nodemanager/v1"
	"io"
	"net/http"
	"os"
	"time"
)

var httpClient *http.Client

type DockerOp struct {
	IsHealthy           bool
	Reason              string
	dockerClient        *client.Client
	UsedExternalPort    map[int64]bool
	SignApi             map[string]string
	ModelsInfo          []*models.ModelInfo
	IsReportModelTaskId map[uint64]bool
	ModelTaskIdChan     chan uint64
}

func init() {
	httpClient = &http.Client{}
}

func NewDockerOp() *DockerOp {
	dockerClient, err := GetDockerClient()
	if err != nil {
		return &DockerOp{
			IsHealthy: false,
			Reason:    fmt.Sprintf("The connect docker client failed reason:%s", err.Error()),
		}
	}
	return &DockerOp{
		IsHealthy:           true,
		Reason:              "",
		dockerClient:        dockerClient,
		SignApi:             make(map[string]string, 0),
		ModelsInfo:          make([]*models.ModelInfo, 0),
		IsReportModelTaskId: make(map[uint64]bool, 0),
		UsedExternalPort:    make(map[int64]bool, 0),
		ModelTaskIdChan:     make(chan uint64, 0),
	}
}

func (d *DockerOp) GetContainerSign(taskMsg *nodemanagerv1.PushTaskMessage, taskRes []byte) []byte {
	reqBody := &models.TaskReq{
		TaskId:     taskMsg.TaskUuid,
		TaskParam:  taskMsg.TaskParam,
		TaskResult: taskRes,
	}
	body, err := json.Marshal(reqBody)
	if err != nil {
		log.Error("Unable to marshal task info: ", err.Error())
		return nil
	}
	taskCmd := &models.TaskCmd{}
	err = json.Unmarshal(bytes.NewBufferString(taskMsg.TaskCmd).Bytes(), taskCmd)
	if err != nil {
		log.Errorf("failed to unmarshal task cmd: %s", err.Error())
		return nil
	}
	// TODO: 请求容器API
	request, err := http.NewRequest("POST", d.SignApi[taskCmd.ImageName], bytes.NewReader(body))
	if err != nil {
		log.Error("New http request failed: ", err)
		return nil
	}
	resp, err := httpClient.Do(request)
	if err != nil {
		log.Error("HTTP request failed: ", err)
		return nil
	}
	all, err := io.ReadAll(resp.Body)
	if err != nil {
		log.Error("Failed to read docker response body:", err)
		return nil
	}
	res := &models.ComputeResult{}
	err = json.Unmarshal(all, res)
	if err != nil {
		log.Error("Failed to parse docker response body")
		return nil
	}
	sign := res.Data
	log.Info("Container sign:", sign)
	return common.Hex2Bytes(sign)
}

func (d *DockerOp) ContainerIsRunning(containerId string) bool {
	inspectContainer := d.inspectContainer(containerId)
	if inspectContainer == nil {
		return false
	}
	return inspectContainer.State.Running
}

func (d *DockerOp) ListContainer() []types.Container {
	ctx, cancel := context.WithTimeout(context.Background(), time.Minute*20)
	defer cancel()
	containers, err := d.dockerClient.ContainerList(ctx, types.ContainerListOptions{})
	if err != nil {
		log.Error("Get container list failed:", err)
		return nil
	}
	for _, c := range containers {
		t := time.Unix(c.Created, 0)
		formattedTime := t.Format("2006-01-02 15:04:05")
		log.Infof("ID: %s, Image: %s, Status: %s , CreateTime: %s \n", c.ID[:10], c.Image, c.Status, formattedTime)
	}
	return containers
}

func (d *DockerOp) CreateAndStartContainer(imageName string, dockerCmd *models.DockerCmd) (string, error) {
	containerId, err := d.CreateContainer(imageName, dockerCmd)
	if err != nil {
		log.Error("Error creating container image failed: ", err)
		return "", err
	}

	// 启动容器
	startContainerIsSuccess := d.StartContainer(containerId)
	if !startContainerIsSuccess {
		log.Error("start container failed:", startContainerIsSuccess)
	}

	//ctx, cancel := context.WithTimeout(context.Background(), time.Second*60)
	//defer cancel()
	//statusCh, errCh := d.dockerClient.ContainerWait(ctx, containerId, container.WaitConditionNotRunning)
	//select {
	//case err := <-errCh:
	//	if err != nil {
	//		panic(err)
	//	}
	//case <-statusCh:
	//	break
	//}
	//
	//out, err := d.dockerClient.ContainerLogs(ctx, containerId, types.ContainerLogsOptions{ShowStdout: true})
	//if err != nil {
	//	panic(err)
	//}
	//
	//_, err = stdcopy.StdCopy(os.Stdout, os.Stderr, out)
	//if err != nil {
	//	log.Error("std out put failed:", err)
	//	return "", err
	//}

	return containerId, nil
}

func (d *DockerOp) CreateContainer(imageName string, dockerCmd *models.DockerCmd) (string, error) {
	ctx, cancel := context.WithTimeout(context.Background(), time.Minute*20)
	defer cancel()
	portBinds := []nat.PortBinding{
		{
			HostIP:   dockerCmd.HostIp,
			HostPort: dockerCmd.HostPort,
		},
	}
	portMap := nat.PortMap{}
	portMap = make(map[nat.Port][]nat.PortBinding, 0)
	portMap[nat.Port(dockerCmd.ContainerPort+"/tcp")] = portBinds

	resp, err := d.dockerClient.ContainerCreate(ctx, &container.Config{
		Image: imageName,
	}, &container.HostConfig{
		PortBindings: portMap,
		AutoRemove:   true, // 容器停止后自动删除
		Resources: container.Resources{
			DeviceRequests: []container.DeviceRequest{
				{
					Driver: "nvidia",
					Count:  -1, // -1 means all GPUs
				},
			},
		},
	}, nil, nil, "")

	if err != nil {
		log.Error("Error creating container image failed: ", err)
		return "", err
	}
	containerId := resp.ID
	log.Info("Container created with ID:", containerId)

	return containerId, nil
}

func (d *DockerOp) StartContainer(containerID string) bool {
	ctx, cancel := context.WithTimeout(context.Background(), time.Minute*20)
	defer cancel()
	// 启动容器
	if err := d.dockerClient.ContainerStart(ctx, containerID, types.ContainerStartOptions{}); err != nil {
		log.Error("Start container failed:", err)
		return false
	}
	log.Info("Container started successfully.")

	return true
}

func (d *DockerOp) StopContainer(containerID string) bool {
	ctx, cancel := context.WithTimeout(context.Background(), time.Minute*20)
	defer cancel()
	// 停止容器（如果正在运行）
	if err := d.dockerClient.ContainerStop(ctx, containerID, container.StopOptions{}); err != nil {
		// 可能容器已经停止或不存在
		log.Info("Error stopping container:", err)
		return false
	}
	log.Info("Container stopped successfully.")
	return true
}

func (d *DockerOp) StopAndDeleteContainer(containerID string) bool {
	// 停止容器（如果正在运行）
	stopContainer := d.StopContainer(containerID)
	if stopContainer {
		d.RmContainer(containerID)
	}
	log.Info("Container stopped successfully.")
	return true
}

func (d *DockerOp) RmContainer(containerID string) {
	ctx, cancel := context.WithTimeout(context.Background(), time.Minute*20)
	defer cancel()
	// 删除容器
	if err := d.dockerClient.ContainerRemove(ctx, containerID, types.ContainerRemoveOptions{}); err != nil {
		panic(err)
	}
	log.Info("Container deleted successfully.")
}

func (d *DockerOp) PsImages() ([]types.ImageSummary, error) {
	ctx, cancel := context.WithTimeout(context.Background(), time.Minute*20)
	defer cancel()
	images, err := d.dockerClient.ImageList(ctx, types.ImageListOptions{})
	if err != nil {
		return nil, err
	}
	for _, c := range images {
		log.Infof("%s %s\n", c.ID[:10], c.ParentID)
	}
	return images, nil
}

func (d *DockerOp) PsImageNameMap() (map[string]bool, error) {
	ctx, cancel := context.WithTimeout(context.Background(), time.Minute*20)
	defer cancel()
	images, err := d.dockerClient.ImageList(ctx, types.ImageListOptions{})
	if err != nil {
		return nil, err
	}
	res := make(map[string]bool, 0)
	for _, image := range images {
		for _, tag := range image.RepoTags {
			res[tag] = true
		}
	}
	return res, nil
}

func (d *DockerOp) PullImage(info *models.ModelInfo) {
	out, err := d.dockerClient.ImagePull(context.Background(), info.ImageName, types.ImagePullOptions{})
	if err != nil {
		log.Errorf("Error pulling image from %s: %v", info.ImageName, err)
		return
	}

	defer func(out io.ReadCloser) {
		err := out.Close()
		if err != nil {
			log.Error("io close failed:", err)
		}
	}(out)

	// 读取拉取镜像的输出
	_, err = io.Copy(os.Stdout, out)
	if err != nil {
		log.Error("pull image info and read failed:", err)
		return
	}

	//_, err = stdcopy.StdCopy(os.Stderr, os.Stdout, out)
	//if err != nil {
	//	log.Error("pull image info and read failed:", err)
	//	return
	//}

	log.Info("Image pulled successfully.")
}

func (d *DockerOp) RmImage(imageId string) {
	ctx, cancel := context.WithTimeout(context.Background(), time.Minute*20)
	defer cancel()
	// 删除镜像
	_, err := d.dockerClient.ImageRemove(ctx, imageId, types.ImageRemoveOptions{})
	if err != nil {
		panic(err)
	}
	log.Info("Image deleted successfully.")
}

func (d *DockerOp) inspectContainer(containerId string) *types.ContainerJSON {
	ctx, cancel := context.WithTimeout(context.Background(), time.Minute*20)
	defer cancel()
	// 容器信息
	containerJson, err := d.dockerClient.ContainerInspect(ctx, containerId)
	if err != nil {
		log.Error("Container inspect failed: ", err)
		return nil
	}
	log.Info("Image deleted successfully.")
	return &containerJson
}
