package db

import (
	"encoding/json"
	"example.com/m/log"
	"example.com/m/models"
	"github.com/syndtr/goleveldb/leveldb"
	"github.com/syndtr/goleveldb/leveldb/iterator"
	"sort"
)

var (
	dbInstance *leveldb.DB
	err        error
	modelKeys  map[string]bool
)

func init() {
	modelKeys = make(map[string]bool, 0)
	// 打开或创建一个LevelDB数据库
	dbInstance, err = leveldb.OpenFile("data/mydb", nil)
	if err != nil {
		log.Error("Leveldb open file failed: ", err)
	} else {
		iter := dbInstance.NewIterator(nil, nil)
		defer iter.Release()
		for iter.Next() {
			key := iter.Key()
			modelKeys[string(key)] = true
		}
		if err := iter.Error(); err != nil {
			log.Fatal(err)
		}
	}
}

func Put(key string, value any) error {
	valueByte, err := json.Marshal(value)
	if err != nil {
		log.Error("Leveldb put data failed:", err)
		return err
	}
	// 存储数据
	err = dbInstance.Put([]byte(key), valueByte, nil)
	if err != nil {
		log.Error("Leveldb put data failed:", err)
		return err
	}
	return nil
}

func NewIterator() (iterator.Iterator, error) {
	// 存储数据
	iteratorRes := dbInstance.NewIterator(nil, nil)
	if iteratorRes.Error() != nil {
		log.Error("Leveldb new iterator failed:", err)
		return nil, err
	}
	return iteratorRes, nil
}

func Get(key string) ([]byte, error) {
	data, err := dbInstance.Get([]byte(key), nil)
	if err != nil {
		log.Error("Leveldb get data failed:", err)
		return nil, err
	}
	//log.WithField("key", key).WithField("value", data).Info("leveldb data")
	return data, nil
}

func PutModel(key string, value any) error {
	valueByte, err := json.Marshal(value)
	if err != nil {
		log.Error("Leveldb put data failed:", err)
		return err
	}
	// 存储数据
	err = dbInstance.Put([]byte(key), valueByte, nil)
	if err != nil {
		log.Error("Leveldb put data failed:", err)
		return err
	}
	modelKeys[key] = true
	return nil
}

func GetAllModels() ([]*models.ModelInfo, error) {
	res := make([]*models.ModelInfo, 0)
	for key := range modelKeys {
		model, _ := GetModel(key)
		if model != nil {
			res = append(res, model)
		}
	}
	return res, nil
}

func GetModel(key string) (*models.ModelInfo, error) {
	data, err := dbInstance.Get([]byte(key), nil)
	if err != nil {
		log.Error("Leveldb get data failed:", err)
		return nil, err
	}
	//log.WithField("key", key).WithField("value", data).Info("leveldb data")
	imageInfo := &models.ModelInfo{}
	err = json.Unmarshal(data, imageInfo)
	if err != nil {
		log.Error("Json decode image data failed:", err)
		return nil, err
	}
	return imageInfo, nil
}

func GetRunningModel() []*models.ModelInfo {
	res := make([]*models.ModelInfo, 0)
	for key := range modelKeys {
		model, _ := GetModel(key)
		if model != nil && model.IsRunning {
			res = append(res, model)
		}
	}
	sort.Slice(res, func(i, j int) bool {
		return res[i].TotalRunCount < res[j].TotalRunCount
	})
	return res
}

func Delete(key []byte) error {
	err := dbInstance.Delete(key, nil)
	if err != nil {
		log.Error("Leveldb del data failed:", err)
		return err
	}
	return nil
}
