package nm

import (
	"context"
	"example.com/m/conf"
	"example.com/m/log"
	"example.com/m/models"
	"example.com/m/operate"
	"example.com/m/validator"
	"fmt"
	nodeManagerV1 "github.com/odysseus/odysseus-protocol/gen/proto/go/nodemanager/v1"
	"google.golang.org/grpc"
	"io"
	"time"
)

type NodeManager struct {
	Info    *nodeManagerV1.NodeManagerInfo
	IsUsed  bool
	IsExist bool
}

// 指定远程 Docker 服务的地址
var (
	isInit                = true
	nodeManagerArr        []*NodeManager
	usedNodeManagerClient []*models.NodeManagerClient
	nodeManagerClientChan chan *models.NodeManagerClient
	nodeManagerMsgChan    chan *nodeManagerV1.ManagerMessage
)

func init() {
	nodeManagerArr = make([]*NodeManager, 0)
	usedNodeManagerClient = make([]*models.NodeManagerClient, 0)
	nodeManagerClientChan = make(chan *models.NodeManagerClient, 0)
	nodeManagerMsgChan = make(chan *nodeManagerV1.ManagerMessage, 1000)
}

func StartMonitor() {
	dockerOp := operate.NewDockerOp()
	if !dockerOp.IsHealthy {
		log.Error("Docker operation is not healthy reason:", dockerOp.Reason)
		panic("Docker client is not healthy")
	}

	go monitorNodeManagerSeed()

	go monitorWorker(dockerOp)

	go monitorModelInfo(dockerOp)

	for isInit {
	}

	connectNodeManagerCount := 0
	for _, manager := range nodeManagerArr {
		if !manager.IsExist {
			continue
		}
		// TODO: 需要对索引进行一定的规则判断，随机选择其中的nodeManager进行链接
		if int64(connectNodeManagerCount) == conf.GetConfig().NodeManagerNum {
			break
		}
		isSuccess := inputNodeManagerChan(manager, nil)
		if !isSuccess {
			panic("Init input node manager chan failed")
		}
		connectNodeManagerCount++
	}
	log.Info("Monitoring node manager client thread start......")

	ticker := time.NewTicker(time.Second * 5)
	for {
		select {
		case <-ticker.C:
			for _, managerClient := range usedNodeManagerClient {
				if !managerClient.GetStatus() {
					manager := getNodeManager(managerClient.Endpoint)
					if manager == nil {
						log.Warn("The managerClient is not exist:", managerClient.Endpoint)
						continue
					}
					isSuccess := false
					if !managerClient.IsDel {
						// TODO: 重试连接三次
						isSuccess = inputNodeManagerChan(manager, managerClient)
					}
					if !isSuccess {
						managerClient.IsDel = true
						unUsedNodeManager := getUnUsedNodeManager()
						if unUsedNodeManager == nil || len(unUsedNodeManager) == 0 {
							break
						}
						for _, nodeManager := range unUsedNodeManager {
							isSuccess := inputNodeManagerChan(nodeManager, nil)
							if !isSuccess {
								break
							}
						}
					}
				}
			}
		}
	}
}

func getUnUsedNodeManager() []*NodeManager {
	res := make([]*NodeManager, 0)
	for _, manager := range nodeManagerArr {
		if !manager.IsUsed && manager.IsExist {
			res = append(res, manager)
		}
	}
	return res
}

func isExistNodeManager(endPoint string) bool {
	for _, manager := range nodeManagerArr {
		if endPoint == manager.Info.Endpoint {
			return true
		}
	}
	return false
}

func monitorNodeManagerSeed() {
	ticker := time.NewTicker(time.Second * 1)
	for {
		select {
		case <-ticker.C:
			seed := conf.GetConfig().NmSeed
			log.Info("Nm seed url:", seed)
			seedServiceClient := operate.ConnNmGrpc(seed)
			if seedServiceClient == nil {
				panic("Dial nm seed service client failed")
			}
			list, err := seedServiceClient.ManagerList(context.Background(), &nodeManagerV1.ManagerListRequest{}, grpc.EmptyCallOption{})
			if err != nil {
				panic("Nm seed seed service is dealing")
			}
			if list.GetManagers() == nil || len(list.GetManagers()) == 0 {
				continue
			}
			for _, node := range list.GetManagers() {
				if isExistNodeManager(node.Endpoint) {
					continue
				}
				nodeManagerArr = append(nodeManagerArr, &NodeManager{Info: node, IsUsed: false, IsExist: true})
			}
			isInit = false
			ticker = time.NewTicker(time.Minute * 10)
		}
	}
}

func GetNodeManagers() []*NodeManager {
	return nodeManagerArr
}

func AddNodeManager(node *nodeManagerV1.NodeManagerInfo) {
	nodeManager := &NodeManager{
		Info:    node,
		IsUsed:  false,
		IsExist: true,
	}
	nodeManagerArr = append(nodeManagerArr, nodeManager)
}

func DelNodeManager(node *nodeManagerV1.NodeManagerInfo) {
	for _, manager := range nodeManagerArr {
		if manager.Info.Endpoint == node.Endpoint {
			manager.IsExist = false
		}
	}
}

// monitorWorker 监听worker
func monitorWorker(op *operate.DockerOp) {
	log.Info("Monitoring worker thread start......")
	for {
		select {
		case managerClient := <-nodeManagerClientChan:
			go func(nodeManager *models.NodeManagerClient) {
				worker, err := nodeManager.Client.RegisterWorker(context.Background(), grpc.EmptyCallOption{})
				if err != nil {
					log.Error("Registration worker failed", err)
					nodeManager.UpdateStatus(false)
					return
				}

				msgRespWorker := NewMsgRespWorker()
				for i := 0; i < 3; i++ {
					go msgRespWorker.SendMsg()
				}

				taskMsgWorker := NewTaskWorker(op)
				taskMsgWorker.HandlerTask(4)

				proofWorker := validator.NewProofWorker()

				// 上报image信息
				go reportModelInfo(nodeManager, worker, msgRespWorker, op)

				// 证明存储
				go proofWorker.ProofStorage()

				// 证明提交
				//go proofWorker.CommitWitness()

				// 处理消息
				for i := 0; i < 5; i++ {
					go handlerMsg(nodeManager, worker, msgRespWorker, taskMsgWorker, proofWorker)
				}

				log.Info("------------------------Start rev msg worker thread------------------------")
				for {
					//if (time.Now().UnixMilli()-nodeManager.GetLastHeartTime())/conf.GetConfig().HeartRespTimeMillis > conf.GetConfig().HeartRespTimeSecond {
					//	nodeManager.UpdateStatus(false)
					//	log.Error("Node manager heartbeat is over")
					//	return
					//}
					rev, err := worker.Recv()
					if err == io.EOF {
						log.Errorf("Node manage not work endpoint:%s", nodeManager.Endpoint)
						nodeManager.UpdateStatus(false)
						params := buildParams(fmt.Sprintf("Node manage not work endpoint:%s", nodeManager.Endpoint))
						msgRespWorker.RegisterMsgResp(nodeManager, worker, GoodbyeResp, params)
						return
					}
					if err != nil {
						log.Error("Rev failed:", err)
						params := buildParams(fmt.Sprintf("Rev failed:%s", err.Error()))
						msgRespWorker.RegisterMsgResp(nodeManager, worker, GoodbyeResp, params)
						return
					}
					log.Info("---------------------Received message---------------------")
					nodeManagerMsgChan <- rev
					log.Info("---------------------Send Received message success---------------------")
					continue
				}
			}(managerClient)
		}
	}
}

func monitorModelInfo(dockerOp *operate.DockerOp) {
	// TODO: 向api端获取model信息
	modelInfo := &models.ModelInfo{
		TaskId:       5,
		User:         "",
		Pwd:          "",
		Repository:   "",
		SignUrl:      "http://192.168.1.120:8888/llm/test/get/sign",
		ImageName:    "demianhjw/aigic:0129",
		DiskSize:     10000,
		MemorySize:   10000,
		IsImageExist: false,
	}
	imageNameMap, err := dockerOp.PsImageNameMap()
	if err != nil {
		log.Error("Docker op ps images failed:", err)
		return
	}
	if !imageNameMap[modelInfo.ImageName] {
		// todo: 判断机器资源是否够用
		isPull := IsResourceEnough(modelInfo)
		// todo： 如果够用
		if isPull {
			go dockerOp.PullImage(modelInfo)
			modelInfo.IsImageExist = true
			dockerOp.ModelTaskIdChan <- modelInfo.TaskId
		}
	} else if !dockerOp.IsReportModelTaskId[modelInfo.TaskId] {
		dockerOp.ModelTaskIdChan <- modelInfo.TaskId
		dockerOp.IsReportModelTaskId[modelInfo.TaskId] = true
	}
	dockerOp.SignApi[modelInfo.ImageName] = modelInfo.SignUrl
	dockerOp.ModelsInfo = append(dockerOp.ModelsInfo, modelInfo)
}

func reportModelInfo(nodeManager *models.NodeManagerClient,
	worker nodeManagerV1.NodeManagerService_RegisterWorkerClient,
	msgRespWorker *RespMsgWorker, dockerOp *operate.DockerOp) {
	for {
		select {
		case taskId := <-dockerOp.ModelTaskIdChan:
			params := buildParams(taskId)
			msgRespWorker.RegisterMsgResp(nodeManager, worker, SubmitResourceMapRes, params)
		}
	}
}

func IsResourceEnough(modelInfo *models.ModelInfo) bool {
	return true
}

// handlerMsg  通过 goroutine 处理Msg
func handlerMsg(nodeManager *models.NodeManagerClient,
	worker nodeManagerV1.NodeManagerService_RegisterWorkerClient,
	msgRespWorker *RespMsgWorker,
	taskMsgWorker *TaskHandler,
	proofWorker *validator.ProofWorker) {
	for {
		select {
		case rev := <-nodeManagerMsgChan:
			{
				heartbeatReq := rev.GetHeartbeatRequest()
				if heartbeatReq != nil {
					nodeManager.UpdateLastHeartTime(int64(heartbeatReq.Timestamp))
					params := buildParams(heartbeatReq.Timestamp)
					msgRespWorker.RegisterMsgResp(nodeManager, worker, HeartbeatResp, params)
					log.Info("-------------Heart beat req:-------------", heartbeatReq)
					continue
				}

				taskMsg := rev.GetPushTaskMessage()
				if taskMsg != nil {
					go func(msgRespWorker *RespMsgWorker,
						taskMsgWorker *TaskHandler, taskMsg *nodeManagerV1.PushTaskMessage) {
						if !taskMsgWorker.DockerOp.IsHealthy {
							params := buildParams(taskMsgWorker.DockerOp.Reason)
							msgRespWorker.RegisterMsgResp(nodeManager, worker, GoodbyeResp, params)
							return
						}
						taskMsgWorker.Wg.Add(1)
						taskMsgWorker.TaskMsg <- taskMsg
						taskMsgWorker.Wg.Wait()
						taskResHeader := taskMsgWorker.TaskRespHeader[taskMsg.TaskUuid]
						taskResBody := taskMsgWorker.TaskRespBody[taskMsg.TaskUuid]
						isSuccess := taskMsgWorker.TaskIsSuccess[taskMsg.TaskUuid]
						containerSign := taskMsgWorker.DockerOp.GetContainerSign(taskMsg, taskResBody)
						if containerSign == nil || len(containerSign) == 0 {
							log.Error("Container signing failed................")
							isSuccess = false
						}
						reqHash, respHash, minerSign := taskMsgWorker.GetMinerSign(taskMsg, taskResBody)
						params := buildParams(taskMsg.TaskUuid, containerSign, minerSign, taskResHeader, taskResBody, isSuccess)
						taskMsgWorker.LruCache.Add(taskMsg.TaskUuid+models.ContainerSign, containerSign)
						taskMsgWorker.LruCache.Add(taskMsg.TaskUuid+models.MinerSign, minerSign)
						taskMsgWorker.LruCache.Add(taskMsg.TaskUuid+models.ReqHash, reqHash)
						taskMsgWorker.LruCache.Add(taskMsg.TaskUuid+models.RespHash, respHash)
						msgRespWorker.RegisterMsgResp(nodeManager, worker, SubmitResultResp, params)
						log.Info("--------------taskMsg--------------:", taskMsg)
					}(msgRespWorker, taskMsgWorker, taskMsg)
					continue
				}

				nmSignMsg := rev.GetProofTaskResult()
				if nmSignMsg != nil {
					containerSign, ok := taskMsgWorker.LruCache.Get(nmSignMsg.TaskUuid + models.ContainerSign)
					if !ok {

					}
					minerSign, ok := taskMsgWorker.LruCache.Get(nmSignMsg.TaskUuid + models.MinerSign)
					if !ok {

					}
					reqHash, ok := taskMsgWorker.LruCache.Get(nmSignMsg.TaskUuid + models.ReqHash)
					if !ok {

					}
					respHash, ok := taskMsgWorker.LruCache.Get(nmSignMsg.TaskUuid + models.RespHash)
					if !ok {

					}
					proofWorker.ProductProof(nmSignMsg.TaskUuid, nmSignMsg.Workload, reqHash.([]byte), respHash.([]byte), containerSign.([]byte), minerSign.([]byte), nmSignMsg.ManagerSignature)
					log.Info(nmSignMsg)
					continue
				}

				deviceMsg := rev.GetDeviceRequest()
				if deviceMsg != nil {
					msgRespWorker.RegisterMsgResp(nodeManager, worker, DeviceInfoResp, nil)
					log.Info(deviceMsg)
					continue
				}

				deviceUsageMsg := rev.GetDeviceUsage()
				if deviceUsageMsg != nil {
					msgRespWorker.RegisterMsgResp(nodeManager, worker, DeviceUsageResp, nil)
					log.Info(deviceUsageMsg)
					continue
				}

				statusReqMsg := rev.GetStatusRequest()
				if statusReqMsg != nil {
					msgRespWorker.RegisterMsgResp(nodeManager, worker, StatusResp, nil)
					log.Info(statusReqMsg)
					continue
				}

				goodByeMsg := rev.GetGoodbyeMessage()
				if goodByeMsg != nil {
					reason := goodByeMsg.GetReason()
					log.Infof("Server endpoint:%s , good bye reason : %s", nodeManager.Endpoint, reason)
					nodeManager.UpdateStatus(false)
					continue
				}
			}
		}
	}
}

func buildParams(params ...interface{}) []interface{} {
	res := make([]interface{}, len(params))
	for i, param := range params {
		res[i] = param
	}
	return res
}

func inputNodeManagerChan(manager *NodeManager, nodeManagerClient *models.NodeManagerClient) bool {
	if nodeManagerClient == nil {
		nodeManagerClient = &models.NodeManagerClient{
			PublicKey:     manager.Info.Publickey,
			Endpoint:      manager.Info.Endpoint,
			Status:        true,
			LastHeartTime: time.Now().UnixMilli(),
		}
	}
	serviceClient := operate.ConnNmGrpc(manager.Info.Endpoint)
	if serviceClient == nil {
		return false
	}
	nodeManagerClient.Status = true
	nodeManagerClient.Client = serviceClient
	nodeManagerClientChan <- nodeManagerClient
	usedNodeManagerClient = append(usedNodeManagerClient, nodeManagerClient)
	manager.IsUsed = true
	return true
}

func getNodeManager(endPoint string) *NodeManager {
	for _, manager := range nodeManagerArr {
		if manager.Info.Endpoint == endPoint {
			return manager
		}
	}
	return nil
}
