package nm

import (
	"context"
	"example.com/m/conf"
	"example.com/m/log"
	"example.com/m/models"
	"example.com/m/operate"
	"example.com/m/utils"
	"example.com/m/validator"
	"fmt"
	nodeManagerV1 "github.com/odysseus/odysseus-protocol/gen/proto/go/nodemanager/v1"
	"google.golang.org/grpc"
	"time"
)

// 指定远程 Docker 服务的地址
var (
	isInit                = false
	nodeManagerArr        []*NodeManager
	usedNodeManagerClient []*models.NodeManagerClient
	nodeManagerClientChan chan *models.NodeManagerClient
	nodeManagerMsgChan    chan *nodeManagerV1.ManagerMessage
)

func init() {
	nodeManagerArr = make([]*NodeManager, 0)
	usedNodeManagerClient = make([]*models.NodeManagerClient, 0)
	nodeManagerClientChan = make(chan *models.NodeManagerClient, 10)
	nodeManagerMsgChan = make(chan *nodeManagerV1.ManagerMessage, 1000)
}

func StartMonitor() {
	dockerOp := operate.NewDockerOp()
	if !dockerOp.IsHealthy {
		log.Error("Docker operation is not healthy reason:", dockerOp.Reason)
		panic("Docker client is not healthy")
	}

	go monitorNodeManagerSeed()

	go monitorWorker(dockerOp)

	go monitorModelInfo(dockerOp)

	for !isInit {
	}

	var connectNodeManagerCount int64 = 0
	selectClientRandomNum := utils.GenerateRandomNumber(conf.GetConfig().SignPrivateKey, conf.GetConfig().NodeManagerNum)
	isSelect := false
	for i := 0; i < len(nodeManagerArr); i++ {
		// TODO: 需要对索引进行一定的规则判断，随机选择其中的nodeManager进行链接
		if connectNodeManagerCount == conf.GetConfig().NodeManagerNum {
			log.Warn("Nothing available node manager..................................")
			break
		}
		randomNum := utils.GenerateRandomNumber(conf.GetConfig().SignPrivateKey, int64(len(nodeManagerArr)))
		manager := nodeManagerArr[randomNum.Int64()]
		if !manager.IsExist {
			log.WithField("endpoint", manager.Info.Endpoint).Warn("node manager is not exist")
			continue
		}
		if !manager.IsUsed {
			if selectClientRandomNum.Int64() == connectNodeManagerCount {
				isSelect = true
			} else {
				isSelect = false
			}
			isSuccess := inputNodeManagerChan(manager, nil, isSelect)
			if !isSuccess {
				log.Warn("Init input node manager chan failed")
				continue
			}
			connectNodeManagerCount++
		}
	}

	ticker := time.NewTicker(time.Second * 5)
	for {
		select {
		case <-ticker.C:
			log.Info("Monitoring node manager client thread start......")
			for _, client := range usedNodeManagerClient {
				log.WithField("Endpoint", client.Endpoint).WithField("LastHeartTime", client.LastHeartTime).WithField("Is Del", client.IsDel).WithField("Status", client.Status).Info("Monitoring node manager client thread")
			}
			for _, managerClient := range usedNodeManagerClient {
				if managerClient.GetStatus() && !managerClient.IsDel {
					sub := time.Now().Sub(managerClient.GetLastHeartTime()).Seconds()
					log.WithField("time(uint seconds)", sub).Info("Main thread monitor nm heartbeat time")
					if int64(sub) > conf.GetConfig().HeartRespTimeSecond {
						managerClient.UpdateStatus(false)
					}
				}
				if !managerClient.GetStatus() && !managerClient.IsDel {
					log.Warn("The Node manager client is failed:", managerClient.Endpoint)
					manager := getNodeManager(managerClient.Endpoint)
					if manager == nil {
						log.Warn("The managerClient is not exist:", managerClient.Endpoint)
						continue
					}
					if !managerClient.IsDel {
						// TODO: 重试连接三次
						isSuccess := inputNodeManagerChan(manager, managerClient, managerClient.IsSelected)
						log.WithField("is success", isSuccess).Warn("Try to connect node manager client:", manager.Info.Endpoint)
						if isSuccess {
							log.Info("Connect node manager client success:", manager.Info.Endpoint)
							managerClient.UpdateLastHeartTime(time.Now())
							if len(dockerOp.ReportTaskIds) > 0 {
								dockerOp.ModelTaskIdIndexesChan <- dockerOp.ReportTaskIds
							}
							continue
						}
					}
					managerClient.IsDel = true
					log.WithField("Endpoint", managerClient.Endpoint).Warn("Node manager client is deleted")
					unUsedNodeManagers := getUnUsedNodeManagers()
					if unUsedNodeManagers == nil || len(unUsedNodeManagers) == 0 {
						log.Warn("There is no node manager available at this time")
						break
					}
					if managerClient.IsSelected {
						for _, client := range usedNodeManagerClient {
							if client.Status && !client.IsDel {
								client.IsSelected = true
								break
							}
						}
					}
					for i := 0; i < len(unUsedNodeManagers); i++ {
						randomNum := utils.GenerateRandomNumber(conf.GetConfig().SignPrivateKey, int64(len(nodeManagerArr)))
						unUsedManager := unUsedNodeManagers[randomNum.Int64()]
						isSuccess := inputNodeManagerChan(unUsedManager, nil, false)
						if !isSuccess {
							log.Warn("Connect unused node manager client error:", manager.Info.Endpoint)
							break
						}
					}
				}
			}
		}
	}
}

// monitorWorker 监听worker
func monitorWorker(op *operate.DockerOp) {
	log.Info("Monitoring worker thread start......")
	for {
		select {
		case managerClient := <-nodeManagerClientChan:
			go func(nodeManager *models.NodeManagerClient) {
				worker, err := nodeManager.Client.RegisterWorker(context.Background(), grpc.EmptyCallOption{})
				if err != nil {
					log.Error("Registration worker failed", err)
					nodeManager.UpdateStatus(false)
					log.Warn("Update nm status is false")
					return
				}

				msgRespWorker := NewMsgRespWorker()
				for i := 0; i < 2; i++ {
					go msgRespWorker.SendMsg()
				}

				taskMsgWorker := NewTaskWorker(op)
				taskMsgWorker.HandlerTask(4)

				proofWorker := validator.NewProofWorker()

				// 主动上报发送设备信息
				go func(isSelect bool) {
					ticker := time.NewTicker(time.Millisecond)
					isSend := false
					for {
						select {
						case <-ticker.C:
							if isSend {
								log.Info("The once-off message is send")
								return
							}
							ticker = time.NewTicker(time.Second * 20)
							msgRespWorker.RegisterMsgResp(nodeManager, worker, RegisterInfoResp, nil)
							time.Sleep(time.Second * 2)
							msgRespWorker.RegisterMsgResp(nodeManager, worker, DeviceInfoResp, nil)
							isSend = true
							log.Info("------------------------Send once-off message ended------------------------")
						}
					}
				}(nodeManager.IsSelected)

				// 上报image信息
				go reportModelInfo(nodeManager, worker, msgRespWorker, op)
				log.Info("Report model info started")

				// 证明存储
				//go proofWorker.ProofStorage()
				//log.Info("Proof storage worker started")

				// 证明提交
				//go proofWorker.CommitWitness()
				//log.Info("Proof commit worker started")

				go handlerStandardTask(nodeManager, worker, msgRespWorker, taskMsgWorker)
				log.Info("Handler standard task worker started")

				// 处理消息
				for i := 0; i < 2; i++ {
					go handlerMsg(nodeManager, worker, msgRespWorker, taskMsgWorker, proofWorker)
				}

				log.Info("------------------------Start rev msg worker thread------------------------")
				for {
					sub := time.Now().Sub(nodeManager.GetLastHeartTime()).Seconds()
					log.WithField("time(uint seconds)", sub).Info("Handler nm msg thread monitor heartbeat time")
					rev, err := worker.Recv()
					if int64(sub) > conf.GetConfig().HeartRespTimeSecond || err != nil {
						log.Error("Rev failed:", err)
						//params := buildParams(fmt.Sprint("Rev failed:", err))
						//msgRespWorker.RegisterMsgResp(nodeManager, worker, GoodbyeResp, params)
						nodeManager.UpdateStatus(false)
						log.Error("Node manager heartbeat is over")
						return
					}
					log.Info("---------------------received message success---------------------")
					nodeManagerMsgChan <- rev
					log.Info("---------------------The message input channel success---------------------")
				}
			}(managerClient)
		}
	}
}

func handlerStandardTask(nodeManager *models.NodeManagerClient,
	worker nodeManagerV1.NodeManagerService_RegisterWorkerClient,
	msgRespWorker *RespMsgWorker,
	taskMsgWorker *TaskHandler) {
	//ticker := time.NewTicker(time.Second * 30)
	ticker := time.NewTicker(time.Minute * 5)
	for {
		select {
		case <-ticker.C:
			{
				if taskMsgWorker.IsExecStandardTask {
					continue
				}
				if !taskMsgWorker.IsExecAiTask {
					msgRespWorker.RegisterMsgResp(nodeManager, worker, FetchStandardTaskResp, nil)
					break
				}
			}
		}
	}
}

// handlerMsg  通过 goroutine 处理Msg
func handlerMsg(nodeManager *models.NodeManagerClient,
	worker nodeManagerV1.NodeManagerService_RegisterWorkerClient,
	msgRespWorker *RespMsgWorker,
	taskMsgWorker *TaskHandler,
	proofWorker *validator.ProofWorker) {
	for {
		select {
		case rev := <-nodeManagerMsgChan:
			{
				if !nodeManager.Status {
					log.Warn("handlerMsg -> node manager is not running")
					return
				}
				heartbeatReq := rev.GetHeartbeatRequest()
				if heartbeatReq != nil {
					nodeManager.UpdateLastHeartTime(time.Now())
					params := buildParams(heartbeatReq.Timestamp)
					msgRespWorker.RegisterMsgResp(nodeManager, worker, HeartbeatResp, params)
					log.Info("-------------Heart beat req:-------------", heartbeatReq)
					continue
				}

				taskMsg := rev.GetPushTaskMessage()
				if taskMsg != nil {
					go func(msgRespWorker *RespMsgWorker,
						taskMsgWorker *TaskHandler, taskMsg *nodeManagerV1.PushTaskMessage) {
						if !taskMsgWorker.DockerOp.IsHealthy {
							//params := buildParams(taskMsgWorker.DockerOp.Reason)
							//msgRespWorker.RegisterMsgResp(nodeManager, worker, GoodbyeResp, params)
							return
						}
						taskMsgWorker.Wg.Add(1)
						taskMsgWorker.TaskMsg <- taskMsg
						taskMsgWorker.Wg.Wait()
						taskMsgWorker.Mutex.Lock()
						taskExecResInterface, _ := taskMsgWorker.LruCache.Get(taskMsg.TaskId)
						taskMsgWorker.Mutex.Unlock()
						//log.WithField("result", taskExecResInterface).Info("lru cache get task result")
						taskExecRes := &models.TaskResult{
							TaskHttpStatusCode: 200,
							TaskRespBody:       nil,
							TaskHttpHeaders:    nil,
							TaskIsSuccess:      false,
							TaskExecTime:       0,
							TaskExecError:      "",
						}
						if taskExecResInterface != nil {
							taskExecRes = taskExecResInterface.(*models.TaskResult)
						}
						isSuccess := taskExecRes.TaskIsSuccess
						containerSign := taskMsgWorker.DockerOp.GetContainerSign(taskMsg, taskExecRes.TaskRespBody)
						if containerSign == nil || len(containerSign) == 0 {
							log.Error("Container signing failed................")
							isSuccess = false
							taskExecRes.TaskExecError = fmt.Sprintf("%s-%s", "Container sign failed", taskExecRes.TaskExecError)
						}
						reqHash, respHash, minerSign := taskMsgWorker.GetMinerSign(taskMsg, taskExecRes.TaskRespBody)
						params := buildParams(taskMsg.TaskId, containerSign, minerSign, taskExecRes, isSuccess)
						taskMsgWorker.Mutex.Lock()
						taskMsgWorker.LruCache.Add(taskMsg.TaskId+models.TaskType, taskMsg.TaskType)
						taskMsgWorker.LruCache.Add(taskMsg.TaskId+models.ContainerSign, containerSign)
						taskMsgWorker.LruCache.Add(taskMsg.TaskId+models.MinerSign, minerSign)
						taskMsgWorker.LruCache.Add(taskMsg.TaskId+models.ReqHash, reqHash)
						taskMsgWorker.LruCache.Add(taskMsg.TaskId+models.RespHash, respHash)
						taskMsgWorker.Mutex.Unlock()
						msgRespWorker.RegisterMsgResp(nodeManager, worker, SubmitResultResp, params)
						log.Info("--------------taskMsg--------------:", taskMsg)
					}(msgRespWorker, taskMsgWorker, taskMsg)
					continue
				}

				nmResultMsg := rev.GetProofTaskResult()
				if nmResultMsg != nil {
					//containerSign, _ := taskMsgWorker.LruCache.Get(nmResultMsg.TaskId + models.ContainerSign)
					//if !ok {
					//	log.Error("taskMsgWorker.LruCache.Get failed: ", nmSignMsg.TaskUuid+models.ContainerSign)
					//}
					//minerSign, _ := taskMsgWorker.LruCache.Get(nmResultMsg.TaskId + models.MinerSign)
					//if !ok {
					//	log.Error("taskMsgWorker.LruCache.Get failed: ", nmSignMsg.TaskUuid+models.MinerSign)
					//}
					//reqHash, _ := taskMsgWorker.LruCache.Get(nmResultMsg.TaskId + models.ReqHash)
					//if !ok {
					//	log.Error("taskMsgWorker.LruCache.Get failed: ", nmSignMsg.TaskUuid+models.ReqHash)
					//}
					//respHash, _ := taskMsgWorker.LruCache.Get(nmResultMsg.TaskId + models.RespHash)
					//if !ok {
					//	log.Error("taskMsgWorker.LruCache.Get failed: ", nmSignMsg.TaskUuid+models.RespHash)
					//}
					//taskType, _ := taskMsgWorker.LruCache.Get(nmResultMsg.TaskId + models.TaskType)
					//proofWorker.ProductProof(nmResultMsg, taskType.(uint64), reqHash.([]byte), respHash.([]byte), containerSign.([]byte), minerSign.([]byte))
					log.WithField("proof", nmResultMsg).Info("Output proof task result")
					continue
				}

				deviceUsageMsg := rev.GetDeviceUsage()
				if deviceUsageMsg != nil {
					msgRespWorker.RegisterMsgResp(nodeManager, worker, DeviceUsageResp, nil)
					log.Info(deviceUsageMsg)
					continue
				}

				nodeInfoMsg := rev.GetNodeInfoRequest()
				if nodeInfoMsg != nil {
					msgRespWorker.RegisterMsgResp(nodeManager, worker, NodeInfoResp, nil)
					log.Info(nodeInfoMsg)
					continue
				}

				statusReqMsg := rev.GetStatusRequest()
				if statusReqMsg != nil {
					msgRespWorker.RegisterMsgResp(nodeManager, worker, StatusResp, nil)
					log.Info(statusReqMsg)
					continue
				}

				goodByeMsg := rev.GetGoodbyeMessage()
				if goodByeMsg != nil {
					reason := goodByeMsg.GetReason()
					log.Infof("Server endpoint:%s , good bye reason : %s", nodeManager.Endpoint, reason)
					nodeManager.UpdateStatus(false)
					log.Warn("Update nm status is false")
					continue
				}
			}
		}
	}
}

func buildParams(params ...interface{}) []interface{} {
	res := make([]interface{}, len(params))
	for i, param := range params {
		res[i] = param
	}
	return res
}
