package conf

import (
	"crypto/ecdsa"
	"example.com/m/log"
	"example.com/m/utils"
	"fmt"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/crypto"
	"net/url"
)

type Config struct {
	SignPub                  string
	DockerServer             string
	BenefitAddress           string
	HeartRespTimeMillis      int64
	ExternalIp               string
	SignPublicAddress        common.Address
	SignPrivateKey           *ecdsa.PrivateKey
	NmSeed                   string `json:"nm_seed" mapstructure:"nm_seed"`
	HeartRespTimeSecond      int64  `json:"heart_response" mapstructure:"heart_response"`
	NodeManagerNum           int64  `json:"node_manager_num" mapstructure:"node_manager_num"`
	ChainID                  int64  `json:"chain_id" mapstructure:"chain_id"`
	ApiUrl                   string `json:"api_url" mapstructure:"api_url"`
	ValidatorUrl             string `json:"validator_url" mapstructure:"validator_url"`
	OssUrl                   string `json:"oss_url" mapstructure:"oss_url"`
	WaitLastTaskExecTime     int64  `json:"wait_last_task_exec_time" mapstructure:"wait_last_task_exec_time"`
	OpSys                    string `json:"op_sys" mapstructure:"op_sys"`
	ReplicateImageNameSuffix string `json:"replicate_image_name_suffix" mapstructure:"replicate_image_name_suffix"`
	IsStopLastContainer      bool   `json:"is_stop_last_container" mapstructure:"is_stop_last_container"`
	DiskUsage                int64  `json:"disk_usage" mapstructure:"disk_usage"`
}

var _cfg *Config = nil

func init() {
	_cfg = &Config{}
}

func GetConfig() *Config {
	return _cfg
}

func (c *Config) SetExternalIp(externalIp string) {
	log.Info("Setting external ip to:", externalIp)
	c.ExternalIp = externalIp
}

func (c *Config) GetExternalIp() string {
	return c.ExternalIp
}

func (c *Config) SetBenefitAddress(addr string) bool {
	isAddr := common.IsHexAddress(addr)
	if isAddr {
		c.BenefitAddress = addr
	}
	return isAddr
}

func (c *Config) SetDockerServerUrl(url string) bool {
	isSuccess, newUrl := checkDockerServer(url)
	log.Info("Setting Docker Server URL:", newUrl)
	if isSuccess {
		c.DockerServer = newUrl
	}
	return isSuccess
}

func (c *Config) SetNmSeed(seed string) {
	c.NmSeed = seed
}

func (c *Config) SetOpSys(sys string) bool {
	if sys != "win" && sys != "mac" && sys != "linux" {
		return false
	}
	c.OpSys = sys
	return true
}

func (c *Config) UpdateFiledInfo() {
	c.HeartRespTimeMillis = c.HeartRespTimeSecond * 60 * 60 * 1000
	prvKey, err := utils.GetPrv()
	if err != nil {
		panic("get prv error or delete keystore after restart")
	}
	c.SignPrivateKey = prvKey
	ecdsaPub := prvKey.PublicKey
	c.SignPub = common.Bytes2Hex(crypto.FromECDSAPub(&ecdsaPub))
	log.Info("PublicKey", c.SignPub)
	publicAddr := crypto.PubkeyToAddress(ecdsaPub)
	log.Info("publicAddr:", publicAddr)
	c.SignPublicAddress = publicAddr
}

func checkDockerServer(rawURL string) (bool, string) {
	if rawURL == "" {
		return true, fmt.Sprintf("tcp://%s:%s", "host.docker.internal", "2375")
	}
	parsedURL, err := url.Parse(rawURL)
	if err != nil {
		log.Error("Error parsing URL:", err)
		return false, ""
	}
	// 提取协议、主机和端口信息
	protocol := parsedURL.Scheme
	if parsedURL.Scheme != "" && protocol != "tcp" {
		protocol = "tcp://"
	}
	ip := "host.docker.internal"
	port := "2375"
	if parsedURL.Port() != "" {
		port = parsedURL.Port()
	}
	if parsedURL.Hostname() != "" {
		ip = parsedURL.Hostname()
	}
	newUrl := fmt.Sprintf("%s://%s:%s", protocol, ip, port)
	return true, newUrl
}
