package largeModel

import (
	"encoding/json"
	"example.com/m/conf"
	"example.com/m/db"
	"example.com/m/log"
	"example.com/m/models"
	"example.com/m/operate"
	"fmt"
	nodemanagerV2 "github.com/odysseus/odysseus-protocol/gen/proto/go/nodemanager/v2"
	"io"
	"net/http"
	"strconv"
	"strings"
	"time"
)

type ModelHandler struct {
	dockerOp *operate.DockerOp
	client   *http.Client
}

func NewModelHandler(dockerOp *operate.DockerOp) *ModelHandler {
	return &ModelHandler{
		dockerOp: dockerOp,
		client:   &http.Client{},
	}
}

func (m *ModelHandler) MonitorModelInfo() {
	ticker := time.NewTicker(time.Second * 1)
	for {
		select {
		case <-ticker.C:
			modelResp, err := m.client.Get(conf.GetConfig().ApiUrl)
			if err != nil {
				log.Error("Error getting model info from client failed:", err)
				continue
			}
			bodyBytes, err := io.ReadAll(modelResp.Body)
			if err != nil {
				log.Error("Error reading model response failed:", err)
				continue
			}
			resp := &models.Resp{}
			err = json.Unmarshal(bodyBytes, resp)
			if err != nil {
				log.Error("Unmarshal model response failed:", err)
				continue
			}
			if resp.Code != http.StatusOK {
				log.Error("Response code :", resp.Code)
				continue
			}
			if resp.Data == nil || len(resp.Data) == 0 {
				log.Warn("Response data is empty")
				continue
			}
			modelInfosResp := resp.Data
			for _, modelInfo := range modelInfosResp {
				if modelInfo.ImageName == "" {
					continue
				}
				modelInfo.ImageName = fmt.Sprintf("%s-%s", modelInfo.ImageName, conf.GetConfig().OpSys)
				split := strings.Split(modelInfo.ImageName, ":")
				if len(split) != 2 {
					continue
				}
				log.WithField("name", modelInfo.ImageName).Info("The image name is already")
				m.dockerOp.SignApi[modelInfo.ImageName] = modelInfo.SignUrl
				model, _ := db.GetModel(modelInfo.ImageName)
				if model != nil {
					model.UpdateFiled(modelInfo)
				}
				err := db.PutModel(modelInfo.ImageName, modelInfo)
				if err != nil {
					log.WithError(err).Error("Put db error")
					continue
				}
			}
			ticker = time.NewTicker(time.Minute * 10)
		}
	}
}

func (m *ModelHandler) GetRpcModelsResp() (*nodemanagerV2.ModelsInfo, error) {
	installedModels := make([]*nodemanagerV2.InstalledModel, 0)
	runningModels := make([]*nodemanagerV2.RunningModel, 0)
	readModels, err := db.GetAllModels()
	if err != nil {
		log.WithError(err).Error("Error reading models")
		return nil, err
	}
	for _, model := range readModels {
		if model.IsInstalled {
			diskSize, err := strconv.ParseInt(model.HardwareRequire.DiskSize, 10, 64)
			if err != nil {
				return nil, err
			}
			model := &nodemanagerV2.InstalledModel{
				ModelId:       strconv.FormatUint(model.TaskId, 10),
				DiskSize:      diskSize,
				InstalledTime: model.SetupTime,
				LastRunTime:   model.LastRunTime,
			}
			installedModels = append(installedModels, model)
		}
		if model.IsRunning {
			model := &nodemanagerV2.RunningModel{
				ModelId: strconv.FormatUint(model.TaskId, 10),
			}
			runningModels = append(runningModels, model)
		}
	}
	res := &nodemanagerV2.ModelsInfo{
		InstalledModels: installedModels,
		RunningModels:   runningModels,
	}
	return res, nil
}

func (m *ModelHandler) MonitorModelStatus() {
	ticker := time.NewTicker(time.Second * 5)
	for {
		select {
		case <-ticker.C:
			{
				imageList, _ := m.dockerOp.PsImages()
				if imageList != nil && len(imageList) > 0 {
					for _, image := range imageList {
						keys := image.RepoTags
						for _, key := range keys {
							model, _ := db.GetModel(key)
							if model != nil && !model.IsInstalled {
								model.SetupTime = time.Now().Unix()
								model.IsInstalled = true
								err := db.PutModel(key, model)
								if err != nil {
									continue
								}
							}
						}
					}
				}
				containerList := m.dockerOp.ListContainer()
				if containerList != nil && len(containerList) > 0 {
					for _, container := range containerList {
						key := container.Image
						model, err := db.GetModel(key)
						if err != nil || model == nil {
							continue
						}
						if container.State == "running" && !model.IsRunning {
							model.ContainerId = container.ID
							model.LastRunTime = time.Now().Unix()
							model.IsRunning = true
							err = db.PutModel(key, model)
							if err != nil {
								continue
							}
						}

					}
				}
			}
		}
	}
}

func (m *ModelHandler) isResourceEnough(modelInfo *models.ModelInfo) bool {
	return true
}

func (m *ModelHandler) checkGpuUsage(modelInfo *models.ModelInfo) bool {
	return false
}

func (m *ModelHandler) checkDiskUsage(modelInfo *models.ModelInfo) bool {
	totalSize, usedSize, availSize, usageSize, err := m.dockerOp.GetDockerInfo()
	if err != nil {
		log.WithError(err).Error("Disk resource is not enough")
		return false
	}
	log.WithField("TotalSize:", totalSize).WithField("UsedSize:", usedSize).WithField("AvailSize:", availSize).Info("The disk info")
	if conf.GetConfig().DiskUsage < usageSize {
		return false
	}
	modelDiskSizeGB, err := strconv.ParseInt(modelInfo.HardwareRequire.DiskSize, 10, 64)
	if err != nil {
		log.Errorf("Error parsing model disk usage failed: %v", err)
		return false
	}
	modelDiskSizeByte := modelDiskSizeGB * 1024 * 1024
	if availSize < modelDiskSizeByte {
		log.Error("The hard drive is running out of space")
		return false
	}
	return true
}
