package nm

import (
	"example.com/m/conf"
	"example.com/m/db"
	"example.com/m/largeModel"
	"example.com/m/log"
	"example.com/m/models"
	"example.com/m/operate"
	"example.com/m/utils"
	"fmt"
	nodemanagerV2 "github.com/odysseus/odysseus-protocol/gen/proto/go/nodemanager/v2"
	"strconv"
	"time"
)

type NodeManagerHandler struct {
	nodeManager   *models.NodeManagerClient
	worker        nodemanagerV2.NodeManagerService_RegisterWorkerClient
	msgRespWorker *RespMsgWorker
	taskMsgWorker *TaskWorker
}

func NewNodeManagerHandler(nodeManager *models.NodeManagerClient, worker nodemanagerV2.NodeManagerService_RegisterWorkerClient, msgRespWorker *RespMsgWorker, taskMsgWorker *TaskWorker) *NodeManagerHandler {
	return &NodeManagerHandler{
		nodeManager:   nodeManager,
		worker:        worker,
		msgRespWorker: msgRespWorker,
		taskMsgWorker: taskMsgWorker,
	}
}

func (n *NodeManagerHandler) DistributionMsgWorker(nodeManagerMsgChan chan *nodemanagerV2.ManagerMessage, modelsHandler *largeModel.ModelHandler) {
	for {
		select {
		case rev := <-nodeManagerMsgChan:
			{
				if !n.nodeManager.Status {
					log.Warn("handlerMsg -> node manager is not running")
					return
				}

				heartbeatReq := rev.GetHeartbeatRequest()
				if heartbeatReq != nil {
					unix := time.Unix(int64(heartbeatReq.Timestamp), 0)
					since := time.Since(unix)
					RunningState.NmDelayTime = since.Milliseconds()
					n.nodeManager.UpdateLastHeartTime(time.Now())
					params := utils.BuildParams(heartbeatReq.Timestamp)
					n.msgRespWorker.RegisterMsgResp(n.nodeManager, n.worker, HeartbeatResp, params)
					log.Info("-------------Heart beat req:-------------", heartbeatReq)
					continue
				}

				taskMsg := rev.GetPushTask()
				if taskMsg != nil {
					go func(msgRespWorker *RespMsgWorker, taskMsgWorker *TaskWorker, taskMsg *nodemanagerV2.PushTaskMessage) {
						isCanExecute, bootUpTime, queueWaitTime, executeTime := taskMsgWorker.GetAckResp(taskMsg)
						ackParams := utils.BuildParams(taskMsg.TaskId, isCanExecute, bootUpTime, queueWaitTime, executeTime)
						msgRespWorker.RegisterMsgResp(n.nodeManager, n.worker, RespTaskAck, ackParams)
						if !isCanExecute {
							log.WithField("taskId", taskMsg.TaskId).Error("The task is not executed")
							return
						}
						taskMsgWorker.Wg.Add(1)
						taskMsgWorker.TaskMsg <- taskMsg
						taskMsgWorker.Wg.Wait()
						taskExecResInterface, _ := taskMsgWorker.LruCache.Get(taskMsg.TaskId)
						//log.WithField("result", taskExecResInterface).Info("lru cache get task result")
						taskExecRes := &models.TaskResult{
							TaskHttpStatusCode: 200,
							TaskRespBody:       nil,
							TaskHttpHeaders:    nil,
							TaskIsSuccess:      false,
							TaskExecTime:       0,
							TaskExecError:      "",
						}
						if taskExecResInterface != nil {
							taskExecRes = taskExecResInterface.(*models.TaskResult)
						}
						isSuccess := taskExecRes.TaskIsSuccess
						containerSign := make([]byte, 0)
						if taskExecRes.TaskRespBody != nil {
							containerSign = taskMsgWorker.DockerOp.GetContainerSign(taskMsg, taskExecRes.TaskRespBody)
							if containerSign == nil || len(containerSign) == 0 {
								log.Error("Container signing failed................")
								isSuccess = false
								taskExecRes.TaskExecError = fmt.Sprintf("%s-%s", "Container sign failed", taskExecRes.TaskExecError)
							}
						} else {
							isSuccess = false
							taskExecRes.TaskExecError = fmt.Sprintf("worker:%s-%s-%s", conf.GetConfig().SignPublicAddress.Hex(), "Task exec error", taskExecRes.TaskExecError)
						}
						reqHash, respHash, minerSign := taskMsgWorker.GetMinerSign(taskMsg, taskExecRes.TaskRespBody)
						taskResultParams := utils.BuildParams(taskMsg.TaskId, containerSign, minerSign, taskExecRes, isSuccess)
						taskMsgWorker.LruCache.Add(taskMsg.TaskId+models.TaskType, taskMsg.TaskType)
						taskMsgWorker.LruCache.Add(taskMsg.TaskId+models.ContainerSign, containerSign)
						taskMsgWorker.LruCache.Add(taskMsg.TaskId+models.MinerSign, minerSign)
						taskMsgWorker.LruCache.Add(taskMsg.TaskId+models.ReqHash, reqHash)
						taskMsgWorker.LruCache.Add(taskMsg.TaskId+models.RespHash, respHash)
						msgRespWorker.RegisterMsgResp(n.nodeManager, n.worker, SubmitResultResp, taskResultParams)
						RunningState.CompletedTaskCount++
						log.Info("Completed task count: ", RunningState.CompletedTaskCount)
						log.Info("--------------taskMsg--------------:", taskMsg)
					}(n.msgRespWorker, n.taskMsgWorker, taskMsg)
					continue
				}

				deviceUsageMsg := rev.GetDeviceUsageRequest()
				if deviceUsageMsg != nil {
					n.msgRespWorker.RegisterMsgResp(n.nodeManager, n.worker, DeviceUsageResp, nil)
					log.Info(deviceUsageMsg)
					continue
				}

				nodeInfoMsg := rev.GetNodeInfoRequest()
				if nodeInfoMsg != nil {
					nodeInfoParam := utils.BuildParams(modelsHandler)
					n.msgRespWorker.RegisterMsgResp(n.nodeManager, n.worker, NodeInfoResp, nodeInfoParam)
					log.Info(nodeInfoMsg)
					continue
				}

				gpuUsageMsg := rev.GetGpuUsageRequest()
				if gpuUsageMsg != nil {
					n.msgRespWorker.RegisterMsgResp(n.nodeManager, n.worker, GpuUsageResp, nil)
					log.Info(gpuUsageMsg)
					continue
				}

				modelListMsg := rev.GetModelListRequest()
				if modelListMsg != nil {
					modelListParam := utils.BuildParams(modelsHandler)
					n.msgRespWorker.RegisterMsgResp(n.nodeManager, n.worker, ModelListResp, modelListParam)
					log.Info(modelListMsg)
					continue
				}

				modelOpMsg := rev.GetModelOperateRequest()
				if modelOpMsg != nil {
					go func(modelOpMsg *nodemanagerV2.ModelOperateRequest, dockerOp *operate.DockerOp) {
						operates := modelOpMsg.GetModelOperates()
						for _, modelOp := range operates {
							go n.MonitorImageOp(modelOp)
							model, err := db.GetModel(modelOp.ImageName)
							if err != nil {
								log.WithError(err).Error("Op model - get model error")
								return
							}
							switch modelOp.Operate {
							case nodemanagerV2.ModelOperateType_INSTALL:
								{
									go dockerOp.PullImage(modelOp.ImageName)
								}
							case nodemanagerV2.ModelOperateType_DELETE:
								{
									if model.ContainerId != "" {
										isRunning := dockerOp.ContainerIsRunning(model.ContainerId)
										if isRunning {
											dockerOp.StopAndDeleteContainer(model.ContainerId)
										}
									}
									go dockerOp.RmImage(modelOp.ImageName)
								}
							case nodemanagerV2.ModelOperateType_RUN:
								{
									envMap := make(map[string]string, 0)
									dockerCmd := &models.DockerCmd{
										EnvMap:   envMap,
										HostIp:   models.ZeroHost,
										HostPort: n.taskMsgWorker.getExternalPort(),
									}
									info := getHardwareInfo()
									gpu := info.GPU
									isMatch := false
									for _, gpuInfo := range gpu {
										if gpuInfo.MemFree > model.GpuRam {
											envMap[models.CudaEnv] = strconv.FormatInt(int64(gpuInfo.Seq), 10)
											isMatch = true
											break
										}
									}
									if !isMatch {

									}
									if isMatch {
										_, err := dockerOp.CreateAndStartContainer(model.ImageName, dockerCmd)
										if err != nil {
											log.WithError(err).Error("Error creating container")
											continue
										}
									}
								}
							case nodemanagerV2.ModelOperateType_STOP:
								{
									if model.ContainerId != "" {
										dockerOp.StopContainer(model.ContainerId)
									}
								}
							}
						}
					}(modelOpMsg, n.taskMsgWorker.DockerOp)
					continue
				}

				deviceInfoMsg := rev.GetDeviceInfoRequest()
				if deviceInfoMsg != nil {
					n.msgRespWorker.RegisterMsgResp(n.nodeManager, n.worker, DeviceInfoResp, nil)
					log.Info(modelListMsg)
					continue
				}

				goodByeMsg := rev.GetGoodbyeMessage()
				if goodByeMsg != nil {
					reason := goodByeMsg.GetReason()
					log.Infof("Server endpoint:%s , good bye reason : %s", n.nodeManager.Endpoint, reason)
					n.nodeManager.UpdateStatus(false)
					log.Warn("Update nm status is false")
					continue
				}
			}
		}
	}
}

func (n *NodeManagerHandler) MonitorImageOp(op *nodemanagerV2.ModelOperate) {
	model, err := db.GetModel(op.ImageName)
	if err != nil {
		log.WithError(err).Error("Op model - get model error")
		return
	}
	model.OpTime = time.Now().Unix()
	ticker := time.NewTicker(time.Second * 2)
	isOp := false
	switch op.Operate {
	case nodemanagerV2.ModelOperateType_INSTALL:
		{
			now := time.Now()
			for {
				select {
				case <-ticker.C:
					if time.Since(now).Seconds() > 36000 || isOp {
						break
					}
					imagesMap, err := n.taskMsgWorker.DockerOp.PsImageNameMap()
					if err != nil {
						log.WithError(err).Error("Ps image name map failed")
						return
					}
					if imagesMap[op.ImageName] {
						isOp = true
						model.IsInstalled = true
						model.SetupTime = time.Now().Unix()
						diskSize, _ := strconv.ParseInt(model.HardwareRequire.DiskSize, 10, 64)
						params := utils.BuildParams(strconv.FormatUint(model.TaskId, 10), diskSize, model.SetupTime, model.LastRunTime)
						n.msgRespWorker.RegisterMsgResp(n.nodeManager, n.worker, AddModelInstalledResp, params)
						return
					}
				}
			}
		}
	case nodemanagerV2.ModelOperateType_DELETE:
		{
			now := time.Now()
			for {
				select {
				case <-ticker.C:
					if time.Since(now).Seconds() > 36000 || isOp {
						break
					}
					imagesMap, err := n.taskMsgWorker.DockerOp.PsImageNameMap()
					if err != nil {
						log.WithError(err).Error("Ps image name map failed")
						return
					}
					if !imagesMap[op.ImageName] {
						isOp = true
						model.IsInstalled = false
						model.IsRunning = false
						model.ContainerId = ""
						params := utils.BuildParams(strconv.FormatUint(model.TaskId, 10))
						n.msgRespWorker.RegisterMsgResp(n.nodeManager, n.worker, DelModelInstalledResp, params)
						return
					}
				}
			}
		}
	case nodemanagerV2.ModelOperateType_RUN:
		{
			now := time.Now()
			for {
				select {
				case <-ticker.C:
					if time.Since(now).Seconds() > 360 || isOp {
						break
					}
					listContainers := n.taskMsgWorker.DockerOp.ListContainer()
					if listContainers != nil && len(listContainers) > 0 {
						for _, container := range listContainers {
							if container.Image == op.ImageName {
								isOp = true
								model.ContainerId = ""
								model.IsRunning = true
								model.LastRunTime = time.Now().Unix()
								params := utils.BuildParams(strconv.FormatUint(model.TaskId, 10), model.GpuSeq, model.GpuRam, model.LastRunTime, model.LastWorkTime, model.TotalRunCount, model.EstimatExeTime)
								n.msgRespWorker.RegisterMsgResp(n.nodeManager, n.worker, AddModelRunningResp, params)
								break
							}
						}
					}
				}
			}
		}
	case nodemanagerV2.ModelOperateType_STOP:
		{
			now := time.Now()
			for {
				select {
				case <-ticker.C:
					if time.Since(now).Seconds() > 360 || isOp {
						break
					}
					listContainers := n.taskMsgWorker.DockerOp.ListContainer()
					if listContainers != nil && len(listContainers) > 0 {
						isFound := false
						for _, container := range listContainers {
							if container.Image == op.ImageName {
								isFound = true
							}
						}
						if !isFound {
							isOp = true
							model.GpuSeq = 999
							model.IsRunning = false
							model.ContainerId = ""
							params := utils.BuildParams(strconv.FormatUint(model.TaskId, 10))
							n.msgRespWorker.RegisterMsgResp(n.nodeManager, n.worker, DelModelRunningResp, params)
							break
						}
					}
				}
			}
		}
	}
	err = db.PutModel(model.ImageName, model)
	if err != nil {
		log.WithError(err).Error("Db put model failed")
	}
}

func (n *NodeManagerHandler) MonitorStandardTaskWorker() {
	//ticker := time.NewTicker(time.Second * 30)
	ticker := time.NewTicker(time.Minute * 5)
	for {
		select {
		case <-ticker.C:
			{
				if n.taskMsgWorker.IsExecStandardTask {
					continue
				}
				if !n.taskMsgWorker.IsExecAiTask {
					n.msgRespWorker.RegisterMsgResp(n.nodeManager, n.worker, FetchStandardTaskResp, nil)
					break
				}
			}
		}
	}
}
