package nm

import (
	"context"
	"example.com/m/conf"
	"example.com/m/log"
	"example.com/m/models"
	"example.com/m/operate"
	"example.com/m/utils"
	"example.com/m/validator"
	"fmt"
	nodeManagerV1 "github.com/odysseus/odysseus-protocol/gen/proto/go/nodemanager/v1"
	"google.golang.org/grpc"
	"io"
	"time"
)

// 指定远程 Docker 服务的地址
var (
	isInit                = true
	nodeManagerArr        []*NodeManager
	usedNodeManagerClient []*models.NodeManagerClient
	nodeManagerClientChan chan *models.NodeManagerClient
	nodeManagerMsgChan    chan *nodeManagerV1.ManagerMessage
)

func init() {
	nodeManagerArr = make([]*NodeManager, 0)
	usedNodeManagerClient = make([]*models.NodeManagerClient, 0)
	nodeManagerClientChan = make(chan *models.NodeManagerClient, 0)
	nodeManagerMsgChan = make(chan *nodeManagerV1.ManagerMessage, 1000)
}

func StartMonitor() {
	dockerOp := operate.NewDockerOp()
	if !dockerOp.IsHealthy {
		log.Error("Docker operation is not healthy reason:", dockerOp.Reason)
		panic("Docker client is not healthy")
	}

	go monitorNodeManagerSeed()

	go monitorWorker(dockerOp)

	go monitorModelInfo(dockerOp)

	for isInit {
	}

	connectNodeManagerCount := 0
	for i := 0; i < len(nodeManagerArr); i++ {
		// TODO: 需要对索引进行一定的规则判断，随机选择其中的nodeManager进行链接
		if int64(connectNodeManagerCount) == conf.GetConfig().NodeManagerNum {
			break
		}
		randomNum := utils.GenerateRandomNumber(conf.GetConfig().SignPrivateKey, int64(len(nodeManagerArr)))
		manager := nodeManagerArr[randomNum.Int64()]
		if !manager.IsExist {
			continue
		}
		if !manager.IsUsed {
			isSuccess := inputNodeManagerChan(manager, nil)
			if !isSuccess {
				panic("Init input node manager chan failed")
			}
			connectNodeManagerCount++
		}
	}

	ticker := time.NewTicker(time.Second * 5)
	for {
		select {
		case <-ticker.C:
			log.Info("Monitoring node manager client thread start......")
			for _, managerClient := range usedNodeManagerClient {
				if !managerClient.GetStatus() {
					log.Warn("The Node manager client is failed:", managerClient.Endpoint)
					manager := getNodeManager(managerClient.Endpoint)
					if manager == nil {
						log.Warn("The managerClient is not exist:", managerClient.Endpoint)
						continue
					}
					if !managerClient.IsDel {
						// TODO: 重试连接三次
						isSuccess := inputNodeManagerChan(manager, managerClient)
						log.Warn("Try to connect node manager client:", manager.Info.Endpoint)
						if isSuccess {
							log.Info("Connect node manager client success:", manager.Info.Endpoint)
							continue
						}
					}
					managerClient.IsDel = true
					unUsedNodeManagers := getUnUsedNodeManagers()
					if unUsedNodeManagers == nil || len(unUsedNodeManagers) == 0 {
						log.Warn("There is no node manager available at this time")
						break
					}
					for i := 0; i < len(unUsedNodeManagers); i++ {
						randomNum := utils.GenerateRandomNumber(conf.GetConfig().SignPrivateKey, int64(len(nodeManagerArr)))
						unUsedManager := unUsedNodeManagers[randomNum.Int64()]
						isSuccess := inputNodeManagerChan(unUsedManager, nil)
						if !isSuccess {
							break
						}
					}
				}
			}
		}
	}
}

// monitorWorker 监听worker
func monitorWorker(op *operate.DockerOp) {
	log.Info("Monitoring worker thread start......")
	for {
		select {
		case managerClient := <-nodeManagerClientChan:
			go func(nodeManager *models.NodeManagerClient) {
				worker, err := nodeManager.Client.RegisterWorker(context.Background(), grpc.EmptyCallOption{})
				if err != nil {
					log.Error("Registration worker failed", err)
					nodeManager.UpdateStatus(false)
					return
				}

				msgRespWorker := NewMsgRespWorker()
				for i := 0; i < 3; i++ {
					go msgRespWorker.SendMsg()
				}

				taskMsgWorker := NewTaskWorker(op)
				taskMsgWorker.HandlerTask(4)

				proofWorker := validator.NewProofWorker()

				// 上报image信息
				go reportModelInfo(nodeManager, worker, msgRespWorker, op)

				// 证明存储
				go proofWorker.ProofStorage()
				log.Info("Proof storage worker started")

				// 证明提交
				go proofWorker.CommitWitness()
				log.Info("Proof commit worker started")

				go handlerStandardTask(nodeManager, worker, msgRespWorker, taskMsgWorker)

				// 处理消息
				for i := 0; i < 5; i++ {
					go handlerMsg(nodeManager, worker, msgRespWorker, taskMsgWorker, proofWorker)
				}

				log.Info("------------------------Start rev msg worker thread------------------------")
				for {
					//if (time.Now().UnixMilli()-nodeManager.GetLastHeartTime())/conf.GetConfig().HeartRespTimeMillis > conf.GetConfig().HeartRespTimeSecond {
					//	nodeManager.UpdateStatus(false)
					//	log.Error("Node manager heartbeat is over")
					//	return
					//}
					rev, err := worker.Recv()
					if err == io.EOF {
						log.Errorf("Node manage not work endpoint:%s", nodeManager.Endpoint)
						params := buildParams(fmt.Sprintf("Node manage not work endpoint:%s", nodeManager.Endpoint))
						msgRespWorker.RegisterMsgResp(nodeManager, worker, GoodbyeResp, params)
						if nodeManager.Status {
							nodeManager.UpdateStatus(false)
						}
						log.Error("Received nm msg failed:%s", err)
						return
					}
					if err != nil {
						log.Error("Rev failed:", err)
						params := buildParams(fmt.Sprintf("Rev failed:%s", err.Error()))
						msgRespWorker.RegisterMsgResp(nodeManager, worker, GoodbyeResp, params)
						if nodeManager.Status {
							nodeManager.UpdateStatus(false)
						}
						return
					}
					log.Info("---------------------Received message---------------------")
					nodeManagerMsgChan <- rev
					log.Info("---------------------Send Received message success---------------------")
					continue
				}
			}(managerClient)
		}
	}
}

func handlerStandardTask(nodeManager *models.NodeManagerClient,
	worker nodeManagerV1.NodeManagerService_RegisterWorkerClient,
	msgRespWorker *RespMsgWorker,
	taskMsgWorker *TaskHandler) {
	ticker := time.NewTicker(time.Second * 30)
	for {
		select {
		case <-ticker.C:
			{
				if taskMsgWorker.IsExecStandardTask {
					continue
				}
				if !taskMsgWorker.IsExecAiTask {
					msgRespWorker.RegisterMsgResp(nodeManager, worker, FetchStandardTaskResp, nil)
					break
				}
			}
		}
	}
}

// handlerMsg  通过 goroutine 处理Msg
func handlerMsg(nodeManager *models.NodeManagerClient,
	worker nodeManagerV1.NodeManagerService_RegisterWorkerClient,
	msgRespWorker *RespMsgWorker,
	taskMsgWorker *TaskHandler,
	proofWorker *validator.ProofWorker) {
	for {
		select {
		case rev := <-nodeManagerMsgChan:
			{
				if !nodeManager.Status {
					return
				}
				heartbeatReq := rev.GetHeartbeatRequest()
				if heartbeatReq != nil {
					nodeManager.UpdateLastHeartTime(int64(heartbeatReq.Timestamp))
					params := buildParams(heartbeatReq.Timestamp)
					msgRespWorker.RegisterMsgResp(nodeManager, worker, HeartbeatResp, params)
					log.Info("-------------Heart beat req:-------------", heartbeatReq)
					continue
				}

				taskMsg := rev.GetPushTaskMessage()
				if taskMsg != nil {
					go func(msgRespWorker *RespMsgWorker,
						taskMsgWorker *TaskHandler, taskMsg *nodeManagerV1.PushTaskMessage) {
						if !taskMsgWorker.DockerOp.IsHealthy {
							params := buildParams(taskMsgWorker.DockerOp.Reason)
							msgRespWorker.RegisterMsgResp(nodeManager, worker, GoodbyeResp, params)
							return
						}
						taskMsgWorker.Wg.Add(1)
						taskMsgWorker.TaskMsg <- taskMsg
						taskMsgWorker.Wg.Wait()
						taskResHeader := taskMsgWorker.TaskRespHeader[taskMsg.TaskId]
						taskResBody := taskMsgWorker.TaskRespBody[taskMsg.TaskId]
						taskResExecTime := taskMsgWorker.TaskExecTime[taskMsg.TaskId]
						isSuccess := taskMsgWorker.TaskIsSuccess[taskMsg.TaskId]
						containerSign := taskMsgWorker.DockerOp.GetContainerSign(taskMsg, taskResBody)
						if containerSign == nil || len(containerSign) == 0 {
							log.Error("Container signing failed................")
							isSuccess = false
						}
						reqHash, respHash, minerSign := taskMsgWorker.GetMinerSign(taskMsg, taskResBody)
						params := buildParams(taskMsg.TaskId, containerSign, minerSign, taskResHeader, taskResBody, taskResExecTime, isSuccess)
						taskMsgWorker.LruCache.Add(taskMsg.TaskId+models.TaskType, taskMsg.TaskType)
						taskMsgWorker.LruCache.Add(taskMsg.TaskId+models.ContainerSign, containerSign)
						taskMsgWorker.LruCache.Add(taskMsg.TaskId+models.MinerSign, minerSign)
						taskMsgWorker.LruCache.Add(taskMsg.TaskId+models.ReqHash, reqHash)
						taskMsgWorker.LruCache.Add(taskMsg.TaskId+models.RespHash, respHash)
						msgRespWorker.RegisterMsgResp(nodeManager, worker, SubmitResultResp, params)
						log.Info("--------------taskMsg--------------:", taskMsg)
					}(msgRespWorker, taskMsgWorker, taskMsg)
					continue
				}

				nmSignMsg := rev.GetProofTaskResult()
				if nmSignMsg != nil {
					containerSign, _ := taskMsgWorker.LruCache.Get(nmSignMsg.TaskId + models.ContainerSign)
					//if !ok {
					//	log.Error("taskMsgWorker.LruCache.Get failed: ", nmSignMsg.TaskUuid+models.ContainerSign)
					//}
					minerSign, _ := taskMsgWorker.LruCache.Get(nmSignMsg.TaskId + models.MinerSign)
					//if !ok {
					//	log.Error("taskMsgWorker.LruCache.Get failed: ", nmSignMsg.TaskUuid+models.MinerSign)
					//}
					reqHash, _ := taskMsgWorker.LruCache.Get(nmSignMsg.TaskId + models.ReqHash)
					//if !ok {
					//	log.Error("taskMsgWorker.LruCache.Get failed: ", nmSignMsg.TaskUuid+models.ReqHash)
					//}
					respHash, _ := taskMsgWorker.LruCache.Get(nmSignMsg.TaskId + models.RespHash)
					//if !ok {
					//	log.Error("taskMsgWorker.LruCache.Get failed: ", nmSignMsg.TaskUuid+models.RespHash)
					//}
					taskType, _ := taskMsgWorker.LruCache.Get(nmSignMsg.TaskId + models.TaskType)
					proofWorker.ProductProof(nmSignMsg.TaskId, nmSignMsg.Workload, taskType.(uint64), reqHash.([]byte), respHash.([]byte), containerSign.([]byte), minerSign.([]byte), nmSignMsg.ManagerSignature)
					log.Info(nmSignMsg)
					continue
				}

				deviceMsg := rev.GetDeviceRequest()
				if deviceMsg != nil {
					msgRespWorker.RegisterMsgResp(nodeManager, worker, DeviceInfoResp, nil)
					log.Info(deviceMsg)
					continue
				}

				deviceUsageMsg := rev.GetDeviceUsage()
				if deviceUsageMsg != nil {
					msgRespWorker.RegisterMsgResp(nodeManager, worker, DeviceUsageResp, nil)
					log.Info(deviceUsageMsg)
					continue
				}

				statusReqMsg := rev.GetStatusRequest()
				if statusReqMsg != nil {
					msgRespWorker.RegisterMsgResp(nodeManager, worker, StatusResp, nil)
					log.Info(statusReqMsg)
					continue
				}

				goodByeMsg := rev.GetGoodbyeMessage()
				if goodByeMsg != nil {
					reason := goodByeMsg.GetReason()
					log.Infof("Server endpoint:%s , good bye reason : %s", nodeManager.Endpoint, reason)
					nodeManager.UpdateStatus(false)
					continue
				}
			}
		}
	}
}

func buildParams(params ...interface{}) []interface{} {
	res := make([]interface{}, len(params))
	for i, param := range params {
		res[i] = param
	}
	return res
}
