package main

import (
	"encoding/json"
	"example.com/m/conf"
	"example.com/m/log"
	"example.com/m/nm"
	"fmt"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/crypto"
	"github.com/spf13/cobra"
	"github.com/spf13/viper"
	"io/ioutil"
	"os"
)

var (
	routineCount uint
	_cfg         *conf.Config = nil
)

// RootCmd represents the base command when called without any subcommands
var RootCmd = &cobra.Command{
	Use:   "Miner",
	Short: "The miner command-line interface",
	Long:  ``,
	Run: func(cmd *cobra.Command, args []string) {
		nm.StartMonitor()
	},
	// Uncomment the following line if your bare application
	// has an action associated with it:
	//	Run: func(cmd *cobra.Command, args []string) { },
}

func init() {
	cobra.OnInitialize(initConfig)
	RootCmd.PersistentFlags().UintVar(&routineCount, "routine", 2, "routine count for corrupt do task")
}

// initConfig reads in conf file and ENV variables if set.
func initConfig() {
	// 设置配置文件的名称（不包含扩展名)
	viper.SetConfigName("config")

	// 设置配置文件的类型
	viper.SetConfigType("json")

	// 设置配置文件所在的目录
	viper.AddConfigPath(".")

	// 读取配置文件
	if err := viper.ReadInConfig(); err != nil {
		fmt.Println("Error reading config file:", err)
		return
	}

	configFilePath := viper.ConfigFileUsed()
	if configFilePath == "" {
		// handle error
		log.Error("config file path is empty")
		panic("config file path is empty")
	}

	data, err := ioutil.ReadFile(configFilePath)
	if err != nil {
		// handle error
		log.Error("Read cfg file error:", err)
		panic("Read cfg file error")
	}
	err = json.Unmarshal(data, &_cfg)
	if err != nil {
		// handle error
		log.Error("Json unmarshal cfg error:", err)
		panic("Json unmarshal cfg error")
	}
	_cfg.HeartRespTimeMillis = _cfg.HeartRespTimeSecond * 60 * 60 * 1000
	prvKey, err := crypto.HexToECDSA(_cfg.SignPrv)
	if err != nil {
		return
	}
	_cfg.SignPrivateKey = prvKey
	ecdsaPub := prvKey.PublicKey
	publicAddr := crypto.PubkeyToAddress(ecdsaPub)
	_cfg.SignPub = common.Bytes2Hex(crypto.FromECDSAPub(&ecdsaPub))
	log.Info(_cfg.SignPub)
	_cfg.SignPublicAddress = publicAddr
}

func Execute() {
	if err := RootCmd.Execute(); err != nil {
		log.Error("root cmd execute failed", err)
		os.Exit(-1)
	}
}
