package validator

import (
	"context"
	"encoding/json"
	"example.com/m/conf"
	"example.com/m/db"
	"example.com/m/log"
	"example.com/m/operate"
	witnessV1 "github.com/odysseus/odysseus-protocol/gen/proto/go/witness/v1"
	"google.golang.org/grpc"
	"math/rand"
	"time"
)

type ProofWorker struct {
	productProofChan chan *witnessV1.Proof
	consumeProofChan chan []*witnessV1.Proof
	isCommitProof    map[string]bool
}

func NewProofWorker() *ProofWorker {
	return &ProofWorker{
		productProofChan: make(chan *witnessV1.Proof, 0),
		consumeProofChan: make(chan []*witnessV1.Proof, 0),
		isCommitProof:    make(map[string]bool, 0),
	}
}

func (p *ProofWorker) ProductProof(taskId string, workLoad uint64, reqHash []byte, respHash []byte, containerSign, minerSign, nmSign []byte) {
	p.productProofChan <- &witnessV1.Proof{
		Workload:           workLoad,
		TaskId:             taskId,
		ReqHash:            reqHash,
		RespHash:           respHash,
		ContainerSignature: containerSign,
		MinerSignature:     minerSign,
		ManagerSignature:   nmSign,
	}
}

func (p *ProofWorker) ProofStorage() {
	go func(productProofChan chan *witnessV1.Proof) {
		for {
			select {
			case proof := <-productProofChan:
				{
					proofByte, err := json.Marshal(proof)
					if err != nil {
						log.Error("Failed to marshal proof: ", err)
						return
					}
					err = db.Put(proof.TaskId, proofByte)
					if err != nil {
						log.Error("leveldb put proof failed: ", err)
						return
					}
					p.isCommitProof[proof.TaskId] = false
				}
			}
		}
	}(p.productProofChan)
	// todo: 需要修改为 Minute
	timer := time.NewTicker(time.Minute)
	defer timer.Stop()
	randomMinute := getRandInt()
	for {
		select {
		case <-timer.C:
			min := time.Now().Minute()
			// 检查是否在指定时间范围内（40-59分钟）
			if min >= 40 && min <= 59 && min == randomMinute {
				randomMinute = getRandInt()
				proofs := make([]*witnessV1.Proof, 0)
				iter, err := db.NewIterator()
				if err != nil {
					log.Error("db new iterator failed: ", err)
					continue
				}
				if iter == nil {
					log.Warn("level db iterator is nil")
					continue
				}
				for iter.Next() {
					proof := &witnessV1.Proof{}
					err := json.Unmarshal(iter.Value(), proof)
					if err != nil {
						log.Error("Error parsing proof from database: ", err)
						return
					}
					if p.isCommitProof[proof.TaskId] {
						continue
					}
					p.isCommitProof[proof.TaskId] = true
					proofs = append(proofs, proof)
					//err = db.Delete(iter.Key())
					//if err != nil {
					//	log.Error("Error deleting proof from database: ", err)
					//	return
					//}
				}
				if len(proofs) > 0 {
					p.consumeProofChan <- proofs
				}
			}
		}
	}
}

func (p *ProofWorker) CommitWitness() {
	validatorClient := operate.ConnValidatorGrpc("192.168.1.180:9431")
	for {
		select {
		case proofs := <-p.consumeProofChan:
			proofsReq := &witnessV1.PushProofRequest{
				Proofs:        proofs,
				MinerAddress:  conf.GetConfig().SignPublicAddress.Hex(),
				RewardAddress: conf.GetConfig().BenefitAddress,
			}
			pushProof, err := validatorClient.PushProof(context.Background(), proofsReq, grpc.EmptyCallOption{})
			if err != nil {
				log.Error("Push proof failed :", err)
				return
			}
			workload := pushProof.GetWorkload()
			log.Info("Push proof response received : %v", workload)
		}
	}
}

func getRandInt() int {
	return rand.Intn(20) + 40
}
