package nm

import (
	"example.com/m/conf"
	"example.com/m/log"
	"example.com/m/models"
	"fmt"
	"github.com/docker/docker/libnetwork/bitmap"
	nodemanagerV1 "github.com/odysseus/odysseus-protocol/gen/proto/go/nodemanager/v1"
	"github.com/shirou/gopsutil/cpu"
	"strconv"
)

type WorkerMsgHandler func(params ...interface{}) *nodemanagerV1.WorkerMessage

type RespMsgHandler struct {
	nodeManager  *models.NodeManagerClient
	workerClient nodemanagerV1.NodeManagerService_RegisterWorkerClient
	handler      WorkerMsgHandler
	params       []interface{}
}

type RespMsgWorker struct {
	MsgPool chan *RespMsgHandler
}

func NewMsgRespWorker() *RespMsgWorker {
	return &RespMsgWorker{
		MsgPool: make(chan *RespMsgHandler, 0),
	}
}

func (o *RespMsgWorker) RegisterMsgResp(nodeManager *models.NodeManagerClient, workerClient nodemanagerV1.NodeManagerService_RegisterWorkerClient, handler WorkerMsgHandler, params []interface{}) {
	o.MsgPool <- &RespMsgHandler{
		nodeManager:  nodeManager,
		workerClient: workerClient,
		handler:      handler,
		params:       params,
	}
	log.Info("----------------add msg response-------------")
}

func (o *RespMsgWorker) SendMsg() {
	for {
		select {
		case pool := <-o.MsgPool:
			{
				workerMsg := pool.handler(pool.params...)
				if workerMsg == nil {
					log.Warn("Send to node manager workerMsg is nil")
					continue
				}
				err := pool.workerClient.SendMsg(workerMsg)
				if err != nil {
					log.Error("Send msg to nm client failed:", err)
					return
				}
				log.Info("Worker client send message successfully")
			}
		}
	}
}

func HeartbeatResp(params ...interface{}) *nodemanagerV1.WorkerMessage {
	log.Info("Heartbeat response received params: ", params)
	serverTimestamp := params[0].(uint64)
	heartRes := &nodemanagerV1.WorkerMessage{
		Message: &nodemanagerV1.WorkerMessage_HeartbeatResponse{
			HeartbeatResponse: &nodemanagerV1.HeartbeatResponse{
				Timestamp: serverTimestamp,
			},
		},
	}
	log.Info("---------------------------------------Send heart beat msg ------------------------------------")
	return heartRes
}

func SubmitResourceMapRes(params ...interface{}) *nodemanagerV1.WorkerMessage {
	log.Info("Submit resource map response received params: ", params)
	taskIdIndexes := params[0].([]uint64)
	taskIdLength := taskIdIndexes[len(taskIdIndexes)-1]
	b := bitmap.New(taskIdLength + 1)
	for i := 0; i < len(taskIdIndexes)-1; i++ {
		taskIdIndex := taskIdIndexes[i]
		err := b.Set(taskIdIndex)
		if err != nil {
			log.WithField("taskId index", taskIdIndex).WithField("error", err).Error("Error setting task id index")
			return nil
		}
	}
	binary, err := b.MarshalBinary()
	if err != nil {
		log.Error("bitmap marshal binary failed with error: ", err)
		return nil
	}
	log.WithField("", binary).Info("Bit map binary byte")
	heartRes := &nodemanagerV1.WorkerMessage{
		Message: &nodemanagerV1.WorkerMessage_ResourceMap{
			ResourceMap: &nodemanagerV1.SubmitResourceMap{
				ResourceMap: binary,
			},
		},
	}
	log.Info("---------------------------------------Send resource map msg ------------------------------------")
	return heartRes
}

func RegisterInfoResp(params ...interface{}) *nodemanagerV1.WorkerMessage {
	log.Info("Register info response received params:", params)
	nodeInfoRes := &nodemanagerV1.WorkerMessage{
		Message: &nodemanagerV1.WorkerMessage_RegisteMessage{
			RegisteMessage: &nodemanagerV1.RegisteMessage{
				DeviceIp:       conf.GetConfig().GetExternalIp(),
				MinerPubkey:    conf.GetConfig().SignPub,
				BenefitAddress: conf.GetConfig().BenefitAddress,
			},
		},
	}
	log.Info("---------------------------------------Send register info msg ------------------------------------")
	return nodeInfoRes
}

func NodeInfoResp(params ...interface{}) *nodemanagerV1.WorkerMessage {
	log.Info("Node info response received params:", params)
	nodeInfoRes := &nodemanagerV1.WorkerMessage{
		Message: &nodemanagerV1.WorkerMessage_NodeInfo{
			NodeInfo: &nodemanagerV1.NodeInfoResponse{
				MinerPubkey:    conf.GetConfig().SignPub,
				BenefitAddress: conf.GetConfig().BenefitAddress,
				DeviceIp:       conf.GetConfig().GetExternalIp(),
			},
		},
	}
	log.Info("---------------------------------------Send node info msg ------------------------------------")
	return nodeInfoRes
}

func DeviceInfoResp(params ...interface{}) *nodemanagerV1.WorkerMessage {
	log.Info("Device info response received params:", params)
	devices := make([]*nodemanagerV1.DeviceInfo, 0)
	cpuInfos, err := cpu.Info()
	if err != nil {
		log.Error("Error getting CPU info: ", err)
	}
	for i, cpuInfo := range cpuInfos {
		cpuInfo := &nodemanagerV1.DeviceInfo{
			DeviceType:  fmt.Sprintf("cpu-%d", i),
			DeviceModel: cpuInfo.ModelName,
			DevicePower: 12,
			DeviceParam: strconv.FormatFloat(cpuInfo.Mhz, 'f', 2, 64),
		}
		devices = append(devices, cpuInfo)
	}

	cpuInfo := &nodemanagerV1.DeviceInfo{
		DeviceType:  "cpu-0",
		DeviceModel: "xl",
		DevicePower: 12,
		DeviceParam: "2150",
	}
	devices = append(devices, cpuInfo)

	gpuInfo := &nodemanagerV1.DeviceInfo{
		DeviceType:  fmt.Sprint("gpu-0"),
		DeviceModel: "Nvidia",
		DevicePower: 12,
		DeviceParam: "1200",
	}
	devices = append(devices, gpuInfo)
	memInfo := &nodemanagerV1.DeviceInfo{
		DeviceType:  fmt.Sprint("mem-0"),
		DeviceModel: "Micron",
		DevicePower: 12,
		DeviceParam: "1200",
	}
	devices = append(devices, memInfo)

	deviceInfoRes := &nodemanagerV1.WorkerMessage{
		Message: &nodemanagerV1.WorkerMessage_DeviceInfo{
			DeviceInfo: &nodemanagerV1.DeviceInfoMessage{
				Devices:         devices,
				DeviceSignature: []byte(""),
			},
		},
	}
	log.Info("---------------------------------------Send device info msg ------------------------------------")
	return deviceInfoRes
}

func DeviceUsageResp(params ...interface{}) *nodemanagerV1.WorkerMessage {
	log.Info("DeviceUsageResp params :", params)
	deviceInfoRes := &nodemanagerV1.WorkerMessage{
		Message: &nodemanagerV1.WorkerMessage_DeviceUsage{
			DeviceUsage: &nodemanagerV1.DeviceUsageResponse{},
		},
	}
	log.Info("---------------------------------------Send device usage msg ------------------------------------")
	return deviceInfoRes
}

func StatusResp(params ...interface{}) *nodemanagerV1.WorkerMessage {
	log.Info("Status resp received params:", params)
	statusRes := &nodemanagerV1.WorkerMessage{
		Message: &nodemanagerV1.WorkerMessage_Status{
			Status: &nodemanagerV1.StatusResponse{
				DeviceStatus: []byte("0"),
			},
		},
	}
	log.Info("---------------------------------------Send device status msg ------------------------------------")
	return statusRes
}

func GoodbyeResp(params ...interface{}) *nodemanagerV1.WorkerMessage {
	log.Info("Goodbye resp received params:", params)
	reason := ""
	if len(params) > 0 {
		reason = params[0].(string)
	}
	goodbyeMsgRes := &nodemanagerV1.WorkerMessage{
		Message: &nodemanagerV1.WorkerMessage_GoodbyeMessage{
			GoodbyeMessage: &nodemanagerV1.GoodbyeMessage{
				Reason: reason,
			},
		},
	}
	log.Info("---------------------------------------Send good bye msg ------------------------------------")
	return goodbyeMsgRes
}

func SubmitResultResp(params ...interface{}) *nodemanagerV1.WorkerMessage {
	//log.Info("Handler task submit result resp received params:", params)
	taskId := params[0].(string)
	containerSign := params[1].([]byte)
	minerSign := params[2].([]byte)
	taskExecResult := params[3].(*models.TaskResult)
	isSuccess := params[4].(bool)
	n := &nodemanagerV1.SubmitTaskResult{
		TaskId:              taskId,
		ContainerSignature:  containerSign,
		MinerSignature:      minerSign,
		TaskExecuteCode:     taskExecResult.TaskHttpStatusCode,
		TaskExecuteError:    taskExecResult.TaskExecError,
		TaskResultHeader:    taskExecResult.TaskHttpHeaders,
		TaskExecuteDuration: uint64(taskExecResult.TaskExecTime),
		IsSuccessed:         isSuccess,
	}
	if taskExecResult.TaskHttpStatusCode == 200 {
		n.TaskResultBody = taskExecResult.TaskRespBody
	}
	submitResultMsgRes := &nodemanagerV1.WorkerMessage{
		Message: &nodemanagerV1.WorkerMessage_SubmitTaskResult{
			SubmitTaskResult: n,
		},
	}
	log.Info("---------------------------------------Send task result msg ------------------------------------")
	return submitResultMsgRes
}

func FetchStandardTaskResp(params ...interface{}) *nodemanagerV1.WorkerMessage {
	//log.Info("Handler task submit result resp received params:", params)
	fetchStandardTaskMsgRes := &nodemanagerV1.WorkerMessage{
		Message: &nodemanagerV1.WorkerMessage_FetchStandardTask{
			FetchStandardTask: &nodemanagerV1.FetchStandardTask{
				TaskType: 998,
			},
		},
	}
	log.Info("---------------------------------------Send fetch standard task msg ------------------------------------")
	return fetchStandardTaskMsgRes
}

func RespTaskAck(params ...interface{}) *nodemanagerV1.WorkerMessage {
	taskId := params[0].(string)
	taskAckMsgRes := &nodemanagerV1.WorkerMessage{
		Message: &nodemanagerV1.WorkerMessage_SubmitTaskAck{
			SubmitTaskAck: &nodemanagerV1.SubmitTaskAck{
				TaskId: taskId,
			},
		},
	}
	log.WithField("taskId", taskId).Info("---------------------------------------Send task ack msg ------------------------------------")
	return taskAckMsgRes
}
