package server

import (
	"context"
	"errors"
	"github.com/IBM/sarama"
	"github.com/gogo/protobuf/proto"
	odysseus "github.com/odysseus/odysseus-protocol/gen/proto/go/base/v1"
	"github.com/odysseus/payment/cachedata"
	"github.com/odysseus/payment/model"
	"github.com/odysseus/scheduler/config"
	"github.com/odysseus/scheduler/utils"
	"github.com/redis/go-redis/v9"
	log "github.com/sirupsen/logrus"
	"strconv"
	"strings"
	"sync"
	"time"
)

type Node struct {
	rdb           *redis.Client
	quit          chan struct{}
	conf          *config.Config
	kafkaProducer sarama.AsyncProducer
	cache         *cachedata.CacheData
	wg            sync.WaitGroup
}

func NewNode() *Node {
	redisConfig := config.GetConfig().Redis
	rdb := utils.NewRedisClient(utils.RedisConnParam{
		Addr:     redisConfig.Addr,
		Password: redisConfig.Password,
		DbIndex:  redisConfig.DbIndex,
	})
	dbconf := config.GetConfig().DbConfig
	pay := cachedata.NewCacheData(context.Background(), cachedata.RedisConnParam{
		Addr:     redisConfig.Addr,
		Password: redisConfig.Password,
		DbIndex:  redisConfig.DbIndex,
	}, model.DbConfig{
		Host:   dbconf.Host,
		Port:   dbconf.Port,
		User:   dbconf.User,
		Passwd: dbconf.Passwd,
		DbName: dbconf.DbName,
	})
	node := &Node{
		cache: pay,
		rdb:   rdb,
		quit:  make(chan struct{}),
		conf:  config.GetConfig(),
	}
	brokers := strings.Split(node.conf.Kafka.Brokers, ";")
	node.kafkaProducer, _ = utils.NewKafkaProducer(brokers)

	return node
}

func (n *Node) Start() error {
	return n.startAllTask()
}

func (n *Node) Stop() {
	close(n.quit)
	n.wg.Wait()
}

func (n *Node) startAllTask() error {
	for i := 0; i < config.GetConfig().Routines; i++ {
		n.wg.Add(1)
		go n.Loop(i)
	}
	return nil
}

func (n *Node) Loop(idx int) {
	defer n.wg.Done()

	defer log.WithField("routine", idx).Info("node loop routine exit")
	// monitor kafka
	taskCh := make(chan *odysseus.TaskContent, 1000)
	ctx, cancel := context.WithCancel(context.Background())
	client, err := n.attachKafkaConsumer(ctx, taskCh)
	if err != nil {
		log.WithError(err).Error("attach kafka consumer failed")
		return
	}
	postResult := func(task *odysseus.TaskContent, result *odysseus.TaskResponse) error {
		d, _ := proto.Marshal(result)
		err := utils.Post(task.TaskCallback, d)
		if err != nil {
			log.WithError(err).Error("post task result failed")
		} else {
			log.WithField("taskid", task.TaskId).Debug("post task result")
		}
		uid, _ := strconv.ParseInt(task.TaskUid, 10, 64)
		fee, _ := strconv.ParseInt(task.TaskFee, 10, 64)
		n.cache.RollbackForFee(uid, fee)

		return err
	}
	postReceipt := func(task *odysseus.TaskContent, result *odysseus.TaskResponse, err error) error {
		receipt := new(odysseus.TaskReceipt)
		receipt.TaskUuid = task.TaskUuid
		receipt.TaskTimestamp = task.TaskTimestamp
		receipt.TaskId = task.TaskId
		receipt.TaskUid = task.TaskUid
		receipt.TaskWorkload = task.TaskWorkload
		receipt.TaskDuration = (time.Now().UnixNano() - int64(task.TaskTimestamp)) / 1000
		receipt.TaskFee = 0
		receipt.TaskOutLen = 0
		receipt.TaskProfitAccount = ""
		receipt.TaskWorkerAccount = ""
		switch err {
		case ErrNoWorker:
			receipt.TaskResult = err.Error()
		case ErrDispatchFailed:
			receipt.TaskResult = err.Error()
		default:
			receipt.TaskResult = "internal error"
		}
		utils.FireTaskReceipt(n.kafkaProducer, receipt, config.GetConfig().Kafka.ReceiptTopic)
		return nil
	}

	for {
		select {
		case <-n.quit:
			cancel()
			client.Close()
			return

		case task := <-taskCh:
			log.WithField("task", task).Info("get task")
			for {
				worker, err := PopWorker(n.rdb)
				if err == ErrNoWorker {
					result := &odysseus.TaskResponse{
						TaskUuid:            task.TaskUuid,
						TaskUid:             task.TaskUid,
						TaskFee:             task.TaskFee,
						TaskIsSucceed:       false,
						TaskError:           err.Error(),
						TaskExecuteDuration: 0,
					}
					postReceipt(task, result, err)
					err = postResult(task, result)
					if err != nil {
						log.WithError(err).Error("post task result failed")
					}
					break
				}

				if err != nil {
					log.WithError(err).Error("pop worker failed")
					continue
				}
				err = DispatchTask(worker, task)
				if err != nil {
					log.WithError(err).Error("dispatch task failed")
					continue
				} else {
					break
				}
			}

		}
	}
}

func (n *Node) attachKafkaConsumer(ctx context.Context, taskCh chan *odysseus.TaskContent) (sarama.ConsumerGroup, error) {
	config := sarama.NewConfig()
	config.Consumer.Return.Errors = true
	config.Consumer.Group.Rebalance.GroupStrategies = []sarama.BalanceStrategy{sarama.NewBalanceStrategyRoundRobin()}
	config.Consumer.Offsets.Initial = sarama.OffsetOldest
	//config.Consumer.Offsets.Initial = sarama.OffsetNewest

	// split broker to list
	brokers := strings.Split(n.conf.Kafka.Brokers, ";")

	client, err := sarama.NewConsumerGroup(brokers, "test", config)
	if err != nil {
		log.WithError(err).Error("creating consumer group client failed")
		return nil, err
	}

	consumeFunc := func(consumer *Consumer) {
		topics := strings.Split(n.conf.Kafka.TaskTopic, ";")
		for {
			if err := client.Consume(ctx, topics, consumer); err != nil {
				if errors.Is(err, sarama.ErrClosedConsumerGroup) {
					return
				}
				log.WithError(err).Error("error from consumer")
			}
			// check if context was cancelled, signaling that the consumer should stop
			if ctx.Err() != nil {
				return
			}
			consumer.ready = make(chan bool)
		}
	}
	go consumeFunc(&Consumer{
		ready:  make(chan bool),
		taskCh: taskCh,
	})

	return client, nil
}

// Consumer represents a Sarama consumer group consumer
type Consumer struct {
	ready  chan bool
	taskCh chan *odysseus.TaskContent
}

// Setup is run at the beginning of a new session, before ConsumeClaim
func (c *Consumer) Setup(sarama.ConsumerGroupSession) error {
	// Mark the consumer as ready
	close(c.ready)
	return nil
}

// Cleanup is run at the end of a session, once all ConsumeClaim goroutines have exited
func (c *Consumer) Cleanup(sarama.ConsumerGroupSession) error {
	return nil
}

// ConsumeClaim must start a consumer loop of ConsumerGroupClaim's Messages().
// Once the Messages() channel is closed, the Handler must finish its processing
// loop and exit.
func (c *Consumer) ConsumeClaim(session sarama.ConsumerGroupSession, claim sarama.ConsumerGroupClaim) error {
	for {
		select {
		case message, ok := <-claim.Messages():
			if !ok {
				log.Printf("message channel was closed")
				return nil
			}
			var task = new(odysseus.TaskContent)
			if err := proto.Unmarshal(message.Value, task); err != nil {
				log.WithError(err).Error("unmarshal task failed")
				continue
			}

			c.taskCh <- task

			session.MarkMessage(message, "")
		case <-session.Context().Done():
			return nil
		}
	}
}
