package main

import (
	"github.com/IBM/sarama"
	"github.com/gogo/protobuf/proto"
	"github.com/google/uuid"
	odysseus "github.com/odysseus/odysseus-protocol/gen/proto/go/base/v1"
	"github.com/odysseus/scheduler/config"
	log "github.com/sirupsen/logrus"
	"strings"
	"time"
)

// NewKafka Create a new KafkaProducer.
func NewKafka(brokers []string) (sarama.AsyncProducer, error) {
	kafkaConfig := sarama.NewConfig()
	kafkaConfig.Producer.RequiredAcks = sarama.WaitForLocal       // Only wait for the leader to ack
	kafkaConfig.Producer.Compression = sarama.CompressionSnappy   // Compress messages
	kafkaConfig.Producer.Flush.Frequency = 500 * time.Millisecond // Flush batches every 500ms
	kafkaConfig.Net.ResolveCanonicalBootstrapServers = false

	producer, err := sarama.NewAsyncProducer(brokers, kafkaConfig)

	if err != nil {
		return nil, err
	}

	go func() {
		for _ = range producer.Errors() {
			//log.Printf("Failed to send log entry to kafka: %v\n", err)
		}
	}()

	return producer, nil
}

func Fire(producer sarama.AsyncProducer, task *odysseus.TaskContent, topic string) error {
	// Check time for partition key
	var partitionKey sarama.StringEncoder
	partitionKey = sarama.StringEncoder(time.Now().Format("2006-01-02"))

	b, err := proto.Marshal(task)
	if err != nil {
		return err
	}
	value := sarama.ByteEncoder(b)

	producer.Input() <- &sarama.ProducerMessage{
		Key:   partitionKey,
		Topic: topic,
		Value: value,
	}

	return nil
}

func produceTask() {
	addr := config.GetConfig().Kafka.Brokers
	borkers := strings.Split(addr, ";")
	producer, err := NewKafka(borkers)
	if err != nil {
		panic(err)
	}
	ticker := time.NewTicker(time.Second * 3)
	defer ticker.Stop()

	for {
		select {
		case <-ticker.C:
			task := makeTask()
			if err := Fire(producer, task, config.GetConfig().Kafka.TaskTopic); err != nil {
				log.WithError(err).Error("fire task failed")
			}
		}
	}
}

func makeTask() *odysseus.TaskContent {
	task := &odysseus.TaskContent{
		TaskId:          uuid.NewString(),
		TaskType:        84,
		TaskKind:        odysseus.TaskKind_ComputeTask,
		TaskCmd:         "{\"api_url\":\"/aigic\",\"docker_cmd\":{\"container_port\":8888},\"image_name\":\"xueqianlu/llm-test:v0.0.1\"}",
		TaskParam:       []byte("{}"),
		TaskTimestamp:   uint64(time.Now().UnixNano() / 1000),
		TaskCallback:    "http://localhost:8080/callback",
		TaskUid:         "51",
		TaskFee:         "0",
		TaskInLen:       10,
		TaskWorkload:    100,
		ContainerPubkey: []byte("0423965c5d5b4c2fa03fe024cfbcd413db25e7c420b73bd5dccb78fada9e22d4848d32952c913ae0cdcc70efa38094dc49c05c6a0cec7e1712557050da7d7bd05a"),
	}
	return task
}
