package server

import (
	"context"
	"errors"
	odysseus "github.com/odysseus/odysseus-protocol/gen/proto/go/base/v1"
	omanager "github.com/odysseus/odysseus-protocol/gen/proto/go/nodemanager/v1"
	"github.com/odysseus/scheduler/config"
	redis "github.com/redis/go-redis/v9"
	log "github.com/sirupsen/logrus"
	"google.golang.org/grpc"
	"google.golang.org/grpc/credentials/insecure"
	"strconv"
)

var (
	maxPriority = 1 // total priority for worker queue
)

var (
	ErrNoWorker       = errors.New("no worker")
	ErrDispatchFailed = errors.New("dispatch to nodemanager failed")
)

type Worker struct {
	workerid string
	priority int
	managers []string
}

func PopWorker(rdb *redis.Client) (Worker, error) {
	for i := 0; i < maxPriority; i++ {
		elem, err := rdb.LPop(context.Background(), config.WORKER_QUEUE_PREFIX+strconv.Itoa(i)).Result()
		if err != nil {
			continue
		}
		managerList, err := rdb.SMembers(context.Background(), config.WORKER_STATUS_PREFIX+elem).Result()
		if err != nil {
			continue
		}
		if len(managerList) == 0 {
			continue
		}
		return Worker{
			workerid: elem,
			priority: i,
			managers: managerList,
		}, nil

	}
	return Worker{}, ErrNoWorker
}

func newManagerClient(endpoint string) (omanager.NodeManagerServiceClient, error) {
	client, err := grpc.Dial(endpoint,
		grpc.WithTransportCredentials(insecure.NewCredentials()),
		grpc.WithDefaultCallOptions(
			grpc.MaxCallRecvMsgSize(1024*1024*1024),
			grpc.MaxCallSendMsgSize(1024*1024*1024)),
	)
	if err != nil {
		return nil, err
	}
	manager := omanager.NewNodeManagerServiceClient(client)
	return manager, nil

}
func DispatchTask(worker Worker, task *odysseus.TaskContent) error {
	for _, manager := range worker.managers {
		client, err := newManagerClient(manager)
		if err != nil {
			log.WithFields(log.Fields{
				"manager": manager,
				"error":   err,
			}).Error("connect to manager failed")
			continue
		}
		_, err = client.DispatchTask(context.Background(), &omanager.DispatchTaskRequest{
			Miner:    worker.workerid,
			TaskData: task,
		})
		if err != nil {
			log.WithFields(log.Fields{
				"manager": manager,
				"error":   err,
			}).Error("dispatch to manager failed")

			continue
		}
		return nil

	}
	return ErrDispatchFailed
}
