package server

import (
	"context"
	"errors"
	"fmt"
	odysseus "github.com/odysseus/odysseus-protocol/gen/proto/go/base/v1"
	omanager "github.com/odysseus/odysseus-protocol/gen/proto/go/nodemanager/v1"
	"github.com/odysseus/scheduler/config"
	redis "github.com/redis/go-redis/v9"
	log "github.com/sirupsen/logrus"
	"google.golang.org/grpc"
	"google.golang.org/grpc/credentials/insecure"
	"strconv"
	"strings"
	"time"
)

var (
	maxPriority = 1 // total priority for worker queue
)

var (
	ErrNoWorker       = errors.New("no worker")
	ErrTimeout        = errors.New("timeout")
	ErrDispatchFailed = errors.New("dispatch to nodemanager failed")
)

type Worker struct {
	workerid string
	addr     string
	nonce    int64
	priority int
	managers []string
}

func PopWorker(ctx context.Context, rdb *redis.Client) (Worker, error) {

	for i := 0; i < maxPriority; i++ {
		for {
			if ctx.Err() != nil {
				return Worker{}, ErrTimeout
			}

			elem, err := rdb.LPop(context.Background(), config.WORKER_QUEUE_PREFIX+strconv.Itoa(i)).Result()
			if err != nil {
				log.WithError(err).Error("lPop worker failed")
				break
			}
			log.WithField("elem", elem).Debug("lPop worker")
			addr, nonce := parseWorkerId(elem)
			managerList, err := rdb.SMembers(context.Background(), workerStatusKey(elem)).Result()
			if err != nil {
				log.WithError(err).Error("get worker status failed")
				continue
			}
			log.WithField("managerList", managerList).Debug("get worker status")
			if len(managerList) == 0 {
				continue
			}
			return Worker{
				workerid: elem,
				addr:     addr,
				nonce:    nonce,
				priority: i,
				managers: managerList,
			}, nil
		}

	}
	return Worker{}, ErrNoWorker
}
func workerStatusKey(wid string) string {
	return fmt.Sprintf("%s_%s", config.WORKER_STATUS_PREFIX, wid)
}
func workerId(w Worker) string {
	return fmt.Sprintf("%s_%d", w.addr, w.nonce)
}

func newManagerClient(endpoint string) (omanager.NodeManagerServiceClient, error) {
	client, err := grpc.Dial(endpoint,
		grpc.WithTransportCredentials(insecure.NewCredentials()),
		grpc.WithDefaultCallOptions(
			grpc.MaxCallRecvMsgSize(1024*1024*1024),
			grpc.MaxCallSendMsgSize(1024*1024*1024)),
	)
	if err != nil {
		return nil, err
	}
	manager := omanager.NewNodeManagerServiceClient(client)
	return manager, nil

}

func parseWorkerId(elem string) (string, int64) {
	split := "_"
	strs := strings.Split(elem, split)
	if len(strs) == 2 {
		addr := strs[0]
		nonce, _ := strconv.ParseInt(strs[1], 10, 64)
		return addr, nonce
	}
	return "", 0
}

func parseWorkerNmValue(nmValue string) (string, int64) {
	split := "#"
	strs := strings.Split(nmValue, split)
	if len(strs) == 2 {
		endpoint := strs[0]
		timestamp, _ := strconv.ParseInt(strs[1], 10, 64)
		return endpoint, timestamp
	}
	return "", 0
}

func (n *Node) DispatchTask(ctx context.Context, worker Worker, task *odysseus.TaskContent) error {
	l := log.WithField("task-id", task.TaskId)
	l.WithFields(log.Fields{
		"worker":      worker.workerid,
		"managerList": worker.managers,
	}).Debug("dispatch task to worker")
	var shouldAddBack = false
	defer func(w Worker) {
		if shouldAddBack {
			// add worker back to redis queue.
			n.rdb.LPush(context.Background(), config.WORKER_QUEUE_PREFIX+strconv.Itoa(w.priority), workerId(w))
			l.WithField("worker", worker.workerid).Debug("add worker back to queue")
		}
	}(worker)

	for _, manager := range worker.managers {
		endpoint, updateTime := parseWorkerNmValue(manager)
		if time.Now().Unix()-updateTime > int64(config.GetConfig().MaxNmUpdateEx) {
			l.WithField("manager", manager).Warn("ignore the manager because of not update tm")
			continue
		}

		client, err := newManagerClient(endpoint)
		if err != nil {
			l.WithFields(log.Fields{
				"manager": endpoint,
				"error":   err,
			}).Error("connect to manager failed")
			continue
		}
		_, err = client.DispatchTask(ctx, &omanager.DispatchTaskRequest{
			Miner:    worker.workerid,
			TaskData: task,
		})
		if err != nil {
			l.WithFields(log.Fields{
				"manager": endpoint,
				"error":   err,
			}).Error("dispatch to manager failed")
			if strings.HasSuffix(err.Error(), "deadline exceeded") {
				shouldAddBack = true
				return ErrTimeout
			}
			continue
		}
		return nil

	}
	return ErrDispatchFailed
}
