package server

import (
	"context"
	"encoding/hex"
	"errors"
	"fmt"
	"github.com/docker/docker/libnetwork/bitmap"
	odysseus "github.com/odysseus/odysseus-protocol/gen/proto/go/base/v1"
	omanager "github.com/odysseus/odysseus-protocol/gen/proto/go/nodemanager/v1"
	"github.com/odysseus/scheduler/config"
	redis "github.com/redis/go-redis/v9"
	log "github.com/sirupsen/logrus"
	"google.golang.org/grpc"
	"google.golang.org/grpc/credentials/insecure"
	"strconv"
	"strings"
	"time"
)

var (
	maxPriority = 2 // total priority for worker queue
)

var (
	ErrNoWorker       = errors.New("no worker")
	ErrTimeout        = errors.New("schedule timeout")
	ErrDispatchFailed = errors.New("dispatch to nodemanager failed")
)

type Worker struct {
	workerid string
	addr     string
	nonce    int64
	priority int
	managers []string
}

func checkWorkerHasResource(rdb *redis.Client, addr string, resource uint64) bool {
	k := workerResourceInfoKey(addr)
	rstr, err := rdb.Get(context.Background(), k).Result()
	if err != nil {
		return false
	}
	data, _ := hex.DecodeString(rstr)
	b := bitmap.New(100000)
	if err := b.UnmarshalBinary(data); err != nil {
		return false
	}
	return b.IsSet(resource)
}

func workerResourceInfoKey(addr string) string {
	return config.WORKER_RESOURCE_INFO_PREFIX + addr
}

func workerStatusKey(addr string, nonce int64) string {
	id := workerId(addr, nonce)
	return fmt.Sprintf("%s_%s", config.WORKER_STATUS_PREFIX, id)
}

func workerId(addr string, nonce int64) string {
	return fmt.Sprintf("%s_%d", addr, nonce)
}

func newManagerClient(endpoint string) (omanager.NodeManagerServiceClient, error) {
	client, err := grpc.Dial(endpoint,
		grpc.WithTransportCredentials(insecure.NewCredentials()),
		grpc.WithDefaultCallOptions(
			grpc.MaxCallRecvMsgSize(1024*1024*1024),
			grpc.MaxCallSendMsgSize(1024*1024*1024)),
	)
	if err != nil {
		return nil, err
	}
	manager := omanager.NewNodeManagerServiceClient(client)
	return manager, nil

}

func parseWorkerId(elem string) (string, int64) {
	split := "_"
	strs := strings.Split(elem, split)
	if len(strs) == 2 {
		addr := strs[0]
		nonceds := strings.Split(strs[1], ":")
		nonce, _ := strconv.ParseInt(nonceds[0], 10, 64)
		return addr, nonce
	}
	return "", 0
}

func parseWorkerNmValue(nmValue string) (string, int64) {
	split := "#"
	strs := strings.Split(nmValue, split)
	if len(strs) == 2 {
		endpoint := strs[0]
		timestamp, _ := strconv.ParseInt(strs[1], 10, 64)
		return endpoint, timestamp
	}
	return "", 0
}

func (n *Node) DispatchTask(ctx context.Context, worker Worker, task *odysseus.TaskContent) error {
	l := log.WithField("task-id", task.TaskId)
	l.WithFields(log.Fields{
		"worker":      worker.workerid,
		"managerList": worker.managers,
	}).Debug("dispatch task to worker")
	var shouldAddBack = false
	defer func(w Worker) {
		if shouldAddBack {
			// add worker back to redis queue.
			n.rdb.LPush(context.Background(), config.WORKER_QUEUE_PREFIX+strconv.Itoa(w.priority), w.workerid)
			l.WithField("worker", worker.workerid).Debug("add worker back to queue")
		}
	}(worker)

	for _, manager := range worker.managers {
		endpoint, updateTime := parseWorkerNmValue(manager)
		if time.Now().Unix()-updateTime > int64(config.GetConfig().MaxNmUpdateEx) {
			l.WithField("manager", manager).Warn("ignore the manager because of not update tm")
			continue
		}

		client, err := newManagerClient(endpoint)
		if err != nil {
			l.WithFields(log.Fields{
				"manager": endpoint,
				"error":   err,
			}).Error("connect to manager failed")
			continue
		}
		_, err = client.DispatchTask(ctx, &omanager.DispatchTaskRequest{
			Miner:    worker.workerid,
			TaskData: task,
		})
		if err != nil {
			l.WithFields(log.Fields{
				"manager": endpoint,
				"error":   err,
			}).Error("dispatch to manager failed")
			if strings.HasSuffix(err.Error(), "deadline exceeded") {
				shouldAddBack = true
				return ErrTimeout
			}
			continue
		}
		return nil

	}
	return ErrDispatchFailed
}
