package workerpoper

import (
	"context"
	"encoding/hex"
	"fmt"
	"github.com/docker/docker/libnetwork/bitmap"
	odysseus "github.com/odysseus/odysseus-protocol/gen/proto/go/base/v1"
	"github.com/odysseus/scheduler/config"
	"github.com/odysseus/scheduler/types"
	"github.com/odysseus/scheduler/utils"
	"github.com/redis/go-redis/v9"
	log "github.com/sirupsen/logrus"
	"strconv"
	"strings"
)

var (
	maxPriority = 2 // total priority for worker queue
)

type poperV1 struct {
	rdb *redis.Client
}

func newPoperV1() (*poperV1, error) {
	redisConfig := config.GetConfig().Redis
	rdb := utils.NewRedisClient(utils.RedisConnParam{
		Addr:     redisConfig.Addr,
		Password: redisConfig.Password,
		DbIndex:  redisConfig.DbIndex,
	})
	return &poperV1{rdb: rdb}, nil
}

func (p *poperV1) CanAddBack() bool {
	return true
}

func (p *poperV1) AddBack(w types.Worker) {
	p.addWorkerBack(w)
}

func (p *poperV1) addWorkerBack(w types.Worker) {
	p.rdb.RPush(context.Background(), config.WORKER_QUEUE_PREFIX+strconv.Itoa(w.Priority), w.Workerid)
	log.WithField("worker", w.Workerid).Debug("add worker back to queue")
}

func (p *poperV1) PopWorker(ctx context.Context, rdb *redis.Client, task *odysseus.TaskContent, ex map[string]bool) (types.Worker, error) {
	// PopWorker implementation
	var checkedWorker = make(map[string]bool)

	for i := 0; i < maxPriority; i++ {
		for {
			if ctx.Err() != nil {
				return types.Worker{}, types.ErrTimeout
			}

			elem, err := rdb.LPop(context.Background(), config.WORKER_QUEUE_PREFIX+strconv.Itoa(i)).Result()
			if err != nil {
				log.WithError(err).Error("lPop worker failed")
				break
			}
			log.WithField("elem", elem).Debug("lPop worker")
			addr, nonce := parseWorkerId(elem)
			log.WithFields(log.Fields{
				"addr":  addr,
				"nonce": nonce,
			}).Debug("parsed worker")
			managerList, err := rdb.SMembers(context.Background(), workerStatusKey(addr, nonce)).Result()
			if err != nil {
				log.WithError(err).Error("get worker status failed")
				continue
			}
			log.WithFields(log.Fields{
				"managerList": managerList,
				"statuskey":   workerStatusKey(addr, nonce),
			}).Debug("get worker status")
			if len(managerList) == 0 {
				continue
			}
			worker := types.Worker{
				Workerid: elem,
				Addr:     addr,
				Nonce:    nonce,
				Priority: i,
				Managers: managerList,
			}
			if false {
				if !checkWorkerHasResource(rdb, worker.Addr, task.TaskType) {
					p.addWorkerBack(worker)
					if checked := checkedWorker[worker.Workerid]; checked {
						break
					} else {
						checkedWorker[worker.Workerid] = true
						continue
					}
				}
			}
			return worker, nil
		}
	}
	return types.Worker{}, types.ErrNoWorker
}

func parseWorkerId(elem string) (string, int64) {
	split := "_"
	strs := strings.Split(elem, split)
	if len(strs) == 2 {
		addr := strs[0]
		nonceds := strings.Split(strs[1], ":")
		nonce, _ := strconv.ParseInt(nonceds[0], 10, 64)
		return addr, nonce
	}
	return "", 0
}

func workerStatusKey(addr string, nonce int64) string {
	id := workerId(addr, nonce)
	return fmt.Sprintf("%s_%s", config.WORKER_STATUS_PREFIX, id)
}

func workerId(addr string, nonce int64) string {
	return fmt.Sprintf("%s_%d", addr, nonce)
}

func checkWorkerHasResource(rdb *redis.Client, addr string, resource uint64) bool {
	k := workerResourceInfoKey(addr)
	rstr, err := rdb.Get(context.Background(), k).Result()
	if err != nil {
		return false
	}
	data, _ := hex.DecodeString(rstr)
	b := bitmap.New(100000)
	if err := b.UnmarshalBinary(data); err != nil {
		return false
	}
	return b.IsSet(resource)
}

func workerResourceInfoKey(addr string) string {
	return config.WORKER_RESOURCE_INFO_PREFIX + addr
}
