package workerpoper

import (
	"context"
	"github.com/odysseus/mogo/operator"
	odysseus "github.com/odysseus/odysseus-protocol/gen/proto/go/base/v1"
	"github.com/odysseus/scheduler/config"
	"github.com/odysseus/scheduler/types"
	"github.com/odysseus/scheduler/utils"
	"github.com/redis/go-redis/v9"
	log "github.com/sirupsen/logrus"
	"go.mongodb.org/mongo-driver/mongo"
	"math/rand"
	"time"
)

type poperV2 struct {
	mogo                    *mongo.Client
	workerInfoOperator      *operator.WorkerInfoOperator
	workerInstalledOperator *operator.WorkerInstalledOperator
	workerRunningOperator   *operator.WorkerRunningOperator
}

func newPoperV2() (*poperV2, error) {
	mogo, err := utils.ConnectMongoDB(config.GetConfig().Mongo.Url, config.GetConfig().Mongo.User, config.GetConfig().Mongo.Passwd)
	if err != nil {
		log.WithError(err).Error("failed to connect mongodb")
		return nil, err
	}
	p := &poperV2{
		mogo:                    mogo,
		workerInfoOperator:      operator.NewDBWorker(mogo, config.GetConfig().Mongo.Database),
		workerInstalledOperator: operator.NewDBWorkerInstalled(mogo, config.GetConfig().Mongo.Database),
		workerRunningOperator:   operator.NewDBWorkerRunning(mogo, config.GetConfig().Mongo.Database),
	}
	return p, nil
}

func (p *poperV2) CanAddBack() bool {
	return false
}

func (p *poperV2) AddBack(w types.Worker) {
	// nothing need to do.
}

func (p *poperV2) PopWorker(ctx context.Context, rdb *redis.Client, task *odysseus.TaskContent, ex map[string]bool) (types.Worker, error) {
	// 1. select from running model worker.
	if ctx.Err() != nil {
		return types.Worker{}, types.ErrTimeout
	}
	t1 := time.Now()
	defer func() {
		log.WithFields(log.Fields{
			"cost": time.Since(t1).Milliseconds(),
		}).Debug("pop worker cost")
	}()
	workers, err := p.workerRunningOperator.FindWorkerByModelId(ctx, int(task.TaskType), 10)
	if err != nil {
		log.WithField("tasktype", task.TaskType).WithError(err).Error("get running model worker failed")
		return types.Worker{}, err
	}
	validWorkers := make([]types.Worker, 0)
	for _, worker := range workers {
		if ex[worker.WorkerId] {
			continue
		}
		addr := worker.WorkerId
		nonce := int64(0)
		managerList, err := rdb.SMembers(context.Background(), workerStatusKey(addr, nonce)).Result()
		if err != nil {
			log.WithError(err).Error("get worker status failed")
			continue
		}
		log.WithFields(log.Fields{
			"managerList": managerList,
			"statuskey":   workerStatusKey(addr, nonce),
		}).Debug("get worker status that found with running model")
		if len(managerList) == 0 {
			continue
		}
		w := types.Worker{
			Workerid: worker.WorkerId,
			Addr:     addr,
			Nonce:    nonce,
			Priority: 0,
			Managers: managerList,
		}
		validWorkers = append(validWorkers, w)
	}
	if len(validWorkers) == 0 {
		// todo: set task.gpu_mem param.
		var taskmem = int64(10)
		// 2. select worker who has installed model.
		workers, err := p.workerInstalledOperator.FindWorkerByModelIdAndGpuMem(ctx, int(task.TaskType), taskmem, 10)
		if err != nil {
			log.WithField("tasktype", task.TaskType).WithError(err).Error("get installed model worker failed")
			return types.Worker{}, err
		}
		for _, worker := range workers {
			if ex[worker.WorkerId] {
				continue
			}
			addr := worker.WorkerId
			nonce := int64(0)
			managerList, err := rdb.SMembers(context.Background(), workerStatusKey(addr, nonce)).Result()
			if err != nil {
				log.WithError(err).Error("get worker status failed")
				continue
			}
			log.WithFields(log.Fields{
				"managerList": managerList,
				"statuskey":   workerStatusKey(addr, nonce),
			}).Debug("get worker status that found with installed model")
			if len(managerList) == 0 {
				continue
			}
			w := types.Worker{
				Workerid: worker.WorkerId,
				Addr:     addr,
				Nonce:    nonce,
				Priority: 0,
				Managers: managerList,
			}
			validWorkers = append(validWorkers, w)
		}
	}
	if len(validWorkers) > 0 {
		// todo: random select an worker
		rdm := rand.Intn(len(validWorkers))
		worker := validWorkers[rdm]

		return worker, nil
	}
	return types.Worker{}, types.ErrNoWorker
}
