package server

import (
	"sdk_api/constant"
	apiModel "sdk_api/model/api"
	"sdk_api/util"
	"strings"

	"github.com/ethereum/go-ethereum/accounts"
	"github.com/ethereum/go-ethereum/common/hexutil"
	"github.com/ethereum/go-ethereum/crypto"
	"github.com/gin-gonic/gin"
	log "github.com/sirupsen/logrus"
	"github.com/tidwall/gjson"
)

func checkUser(c *gin.Context) {
	req := &apiModel.CheckUserRequest{}
	if err := c.ShouldBindJSON(req); err != nil {
		withError(constant.InvalidParam)
		return
	}
	switch req.Platform {
	case constant.PlatformTelegram:
		ok, userId := util.VerifyInitData(req.InitData, "5000822884:AAHtZnIGjV1yYMfN6s_pAzoiZWDf5JWWTko")
		if !ok {
			c.JSON(200, withError("invalid initData"))
			return
		}

		ok, uid, keystore, err := srv.CheckUser(constant.PlatformTelegram, userId)
		if err != nil {
			c.JSON(200, withError(constant.InternalError))
			return
		}

		token := util.GenerateJWT(uid, constant.PlatformTelegram, userId)

		resp := &apiModel.CheckUserResponse{
			IsNewUser: !ok,
			Keystore:  keystore,
			Token:     token,
		}
		c.JSON(200, withSuccess(resp))
		return

	case constant.PlatformFingerprint:
		userId := req.VisitorID

		ok, uid, keystore, err := srv.CheckUser(constant.PlatformFingerprint, userId)
		if err != nil {
			c.JSON(200, withError(constant.InternalError))
			return
		}

		token := util.GenerateJWT(uid, constant.PlatformFingerprint, userId)

		resp := &apiModel.CheckUserResponse{
			IsNewUser: !ok,
			Keystore:  keystore,
			Token:     token,
		}

		c.JSON(200, withSuccess(resp))

	default:
		c.JSON(200, withError(constant.UnsupportedPlatform))
		return
	}
}

func createUser(c *gin.Context) {
	req := &apiModel.CreateUserRequest{}
	if err := c.ShouldBindJSON(req); err != nil {
		c.JSON(200, withError(constant.InvalidParam))
		return
	}

	uid := c.GetString("jwt-uid")

	address := gjson.Get(req.Keystore, "address").String()
	binSignature, err := hexutil.Decode(req.Signature)
	if err != nil || len(binSignature) < 65 {
		c.JSON(200, withError("invalid signature"))
		return
	}
	binSignature[64] -= 27
	ecdsaPub, err := crypto.SigToPub(accounts.TextHash([]byte(req.Keystore)), binSignature)
	if err != nil {
		c.JSON(200, withError("invalid signature"))
		return
	}
	addr := crypto.PubkeyToAddress(*ecdsaPub)
	log.Debugln(addr.Hex())
	if strings.ToLower(addr.Hex()[2:]) != address {
		c.JSON(200, withError("invalid signature"))
		return
	}

	ok, err := srv.SetKeystore(uid, strings.ToLower(addr.Hex()), req.Keystore)
	if err != nil {
		c.JSON(200, withError(constant.InternalError))
		return
	}

	if !ok {
		c.JSON(200, withError("keystore already exist"))
		return
	}
	c.JSON(200, withSuccess(""))
}
