package exec

import (
	"math/big"
	"sdk_api/config"
	"sdk_api/constant"
	"sdk_api/dao"
	"sync"
	"time"

	"github.com/ethereum/go-ethereum/common"
	log "github.com/sirupsen/logrus"
)

type Executor struct {
	c      *config.Config
	d      *dao.Dao
	nonces map[common.Address]uint64
	sync.Mutex
}

func NewExecutor(_c *config.Config, _d *dao.Dao) *Executor {
	t := &Executor{
		c: _c,
		d: _d,
	}
	nonce, err := _d.GetNonce(common.HexToAddress(_c.Chain.SenderAddress))
	if err != nil {
		panic(err)
	}
	log.WithFields(log.Fields{"nonce": nonce, "addr": _c.Chain.SenderAddress}).Info("get nonce from chain")
	t.nonces = make(map[common.Address]uint64)
	t.nonces[common.HexToAddress(_c.Chain.SenderAddress)] = nonce
	return t
}

func (e *Executor) Start() {
	e.ProcessReceipt(true)

	go e.ProcessTransaction()
	go e.ProcessReceipt(false)

}

func (e *Executor) ProcessReceipt(sync bool) {
	// sync:true 启动时运行
	log.WithField("sync", sync).Info("start process receipt")
	ticker := time.NewTicker(200 * time.Millisecond)
	defer ticker.Stop()
	for {
		txHashes, err := e.d.GetTxHashesByStatus(constant.TransactionNotExecuted)
		if err != nil {
			log.WithError(err).Error("get txHashes failed")
			time.Sleep(time.Millisecond * 500)
			continue
		}
		if len(txHashes) == 0 && sync {
			log.Info("no tx to process receipt, stop")
			return
		}

		for _, txHash := range txHashes {
			var status int
			timeout := 30 * time.Second
			if sync {
				timeout = 5 * time.Second
			}
			receipt, err := e.d.WaitForReceipt(common.HexToHash(txHash), timeout)
			if err != nil {
				log.WithFields(log.Fields{
					"txHash": txHash,
					"err":    err,
					"sync":   sync,
				}).Error("wait receipt failed")
				status = constant.TransactionReceiptError
			} else if receipt == nil {
				status = constant.TransactionNotFound
			} else if receipt.Status == 1 {
				status = constant.TransactionSuccessful
			} else if receipt.Status == 0 {
				status = constant.TransactionFailed
			}

			log.WithFields(log.Fields{"txHash": txHash, "status": status}).Info("get tx receipt")

			err = e.d.UpdateTaskStatus(txHash, txHash, status)
			if err != nil {
				log.WithFields(log.Fields{
					"txHash": txHash,
					"err":    err,
					"sync":   sync,
				}).Error("update task status failed")
			}
		}
		<-ticker.C
	}
}

func (e *Executor) ProcessTransaction() {
	log.Info("start process transaction")
	ticker := time.NewTicker(200 * time.Millisecond)
	defer ticker.Stop()
	for {
		tasks, err := e.d.GetNotExecutedTasks()
		if err != nil {
			log.WithError(err).Error("get not executed tasks failed")
			time.Sleep(time.Millisecond * 500)
			continue
		}

		for _, task := range tasks {
			e.Lock()
			nonce := e.nonces[common.HexToAddress(e.c.Chain.SenderAddress)]
			e.Unlock()
			val, _ := new(big.Int).SetString(task.Value, 10)
			txHash, sent, err := e.d.BroadcastTx(
				common.HexToAddress(task.ToAddress),
				nonce,
				val,
				common.FromHex(task.Calldata),
			)

			if sent {
				e.Lock()
				e.nonces[common.HexToAddress(e.c.Chain.SenderAddress)]++
				e.Unlock()
			}

			// 交易成功
			if err == nil {
				err = e.d.UpdateTaskStatus(task.TaskId, txHash.String(), constant.TransactionPending)
				if err != nil {
					log.WithFields(log.Fields{"taskId": task.TaskId, "err": err}).Error("update task status failed")
					continue
				}
				log.WithFields(log.Fields{"taskId": task.TaskId, "action": task.Action, "txHash": txHash}).Info("broadcast tx success")
				continue
			}

			// 交易失败
			log.WithFields(log.Fields{"taskId": task.TaskId, "err": err}).Error("broadcast tx failed")
			err = e.d.UpdateTaskStatus(task.TaskId, "", constant.TransactionBroadcastError)
			if err != nil {
				log.WithFields(log.Fields{"taskId": task.TaskId, "err": err}).Error("update task status failed")
			}
		}

		<-ticker.C
	}
}
