package gassender

import (
	"context"
	"crypto/ecdsa"
	"math/big"
	"time"

	"github.com/ethereum/go-ethereum"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/core/types"
	"github.com/ethereum/go-ethereum/crypto"
	"github.com/ethereum/go-ethereum/ethclient"
	log "github.com/sirupsen/logrus"
)

type GasSender struct {
	privateKey *ecdsa.PrivateKey
	client     *ethclient.Client
	taskCh     chan *gasTask
	chainId    *big.Int
}

type gasTask struct {
	dest  common.Address
	value *big.Int
}

func NewGasSender(rpc, privateKey string) (*GasSender, error) {
	ecdsaKey, err := crypto.HexToECDSA(common.Bytes2Hex(common.FromHex(privateKey)))
	if err != nil {
		return nil, err
	}

	log.WithField("address", crypto.PubkeyToAddress(ecdsaKey.PublicKey)).Info("aon gas sender address")

	client, err := ethclient.Dial(rpc)
	if err != nil {
		return nil, err
	}

	chainId, err := client.ChainID(context.Background())
	if err != nil {
		return nil, err
	}

	return &GasSender{
		privateKey: ecdsaKey,
		client:     client,
		taskCh:     make(chan *gasTask, 32),
		chainId:    chainId,
	}, nil

}

func (gs *GasSender) Run() {
	go func() {
		for {
			select {
			case task := <-gs.taskCh:
				gs.sendTx(task)
			}
		}
	}()
}

func (gs *GasSender) SendAONGas(dest string, amount int) {
	addr := common.HexToAddress(dest)
	value := new(big.Int).Mul(big.NewInt(int64(amount)), big.NewInt(1000000000000000000))
	gs.taskCh <- &gasTask{
		dest:  addr,
		value: value,
	}
}

func (gs *GasSender) sendTx(task *gasTask) {
	log.WithFields(log.Fields{
		"address": task.dest,
		"value":   task.value,
	}).Info("new send gas task")
	nonce, err := gs.client.NonceAt(context.Background(), crypto.PubkeyToAddress(gs.privateKey.PublicKey), nil)
	if err != nil {
		log.WithError(err).Error("get nonce failed")
		return
	}

	tx := &types.LegacyTx{
		Nonce:    nonce,
		GasPrice: big.NewInt(1000000000),
		Gas:      21000,
		To:       &task.dest,
		Value:    task.value,
	}

	signer := types.NewEIP155Signer(gs.chainId)
	signedTx, err := types.SignNewTx(gs.privateKey, signer, tx)
	if err != nil {
		log.WithError(err).Error("sign tx failed")
		return
	}

	err = gs.client.SendTransaction(context.Background(), signedTx)
	if err != nil {
		log.WithError(err).Error("send tx failed")
		return
	}

	txLog := log.WithField("txHash", signedTx.Hash().Hex())
	txLog.Info("tx broadcasted")
	for i := 0; i < 5; i++ {
		time.Sleep(time.Second * 2)
		receipt, err := gs.client.TransactionReceipt(context.Background(), signedTx.Hash())
		if err != nil && err == ethereum.NotFound {
			txLog.Info("tx receipt not found, retrying...")
			continue
		}
		if err != nil {
			txLog.WithError(err).Error("send gas tx failed")
			return
		}

		if receipt.Status != 1 {
			txLog.Error("send gas tx failed")
			return
		}
		txLog.Info("send gas tx confirmed")
		return
	}
	txLog.Error("tx receipt not found, timeout")
}
