package gassender

import (
	"context"
	"crypto/ecdsa"
	"math/big"
	"sdk_api/contract/aonUser"
	"time"

	"github.com/ethereum/go-ethereum"
	"github.com/ethereum/go-ethereum/accounts/abi/bind"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/core/types"
	"github.com/ethereum/go-ethereum/crypto"
	"github.com/ethereum/go-ethereum/ethclient"
	log "github.com/sirupsen/logrus"
)

type GasSender struct {
	privateKey *ecdsa.PrivateKey
	client     *ethclient.Client
	taskCh     chan *gasTask
	chainId    *big.Int
	aonUser    *aonUser.AonUser
}

type gasTask struct {
	dest      common.Address
	user      common.Address
	value     *big.Int
	userId    string
	inviterId string
	method    string
}

func NewGasSender(rpc, privateKey, aonUserContract string) (*GasSender, error) {
	ecdsaKey, err := crypto.HexToECDSA(common.Bytes2Hex(common.FromHex(privateKey)))
	if err != nil {
		return nil, err
	}

	log.WithField("address", crypto.PubkeyToAddress(ecdsaKey.PublicKey)).Info("aon gas sender address")

	client, err := ethclient.Dial(rpc)
	if err != nil {
		return nil, err
	}

	chainId, err := client.ChainID(context.Background())
	if err != nil {
		return nil, err
	}

	ca, err := aonUser.NewAonUser(common.HexToAddress(aonUserContract), client)
	if err != nil {
		return nil, err
	}

	log.WithFields(log.Fields{
		"caller":       crypto.PubkeyToAddress(ecdsaKey.PublicKey),
		"userContract": common.HexToAddress(aonUserContract),
	}).Info("init gas sender")

	return &GasSender{
		privateKey: ecdsaKey,
		client:     client,
		taskCh:     make(chan *gasTask, 32),
		chainId:    chainId,
		aonUser:    ca,
	}, nil

}

func (gs *GasSender) Run() {
	go func() {
		for {
			select {
			case task := <-gs.taskCh:
				switch task.method {
				case "sendGas":
					gs.sendGas(task)
				case "aonLogin":
					gs.login(task)
				}
			}
		}
	}()
}

func (gs *GasSender) SendAONGas(dest string, amount int) {
	addr := common.HexToAddress(dest)
	value := new(big.Int).Mul(big.NewInt(int64(amount)), big.NewInt(1000000000000000000))
	gs.taskCh <- &gasTask{
		dest:   addr,
		value:  value,
		method: "sendGas",
	}
}

func (gs *GasSender) AonLogin(address common.Address, userId, inviter string) {
	gs.taskCh <- &gasTask{
		user:      address,
		value:     big.NewInt(0),
		userId:    userId,
		inviterId: inviter,
		method:    "aonLogin",
	}
}

func (gs *GasSender) sendGas(task *gasTask) {
	log.WithFields(log.Fields{
		"address": task.dest,
		"value":   task.value,
	}).Info("new send gas task")
	nonce, err := gs.client.NonceAt(context.Background(), crypto.PubkeyToAddress(gs.privateKey.PublicKey), nil)
	if err != nil {
		log.WithError(err).Error("get nonce failed")
		return
	}

	tx := &types.LegacyTx{
		Nonce:    nonce,
		GasPrice: big.NewInt(1000000000),
		Gas:      21000,
		To:       &task.dest,
		Value:    task.value,
	}

	signer := types.NewEIP155Signer(gs.chainId)
	signedTx, err := types.SignNewTx(gs.privateKey, signer, tx)
	if err != nil {
		log.WithError(err).Error("sign tx failed")
		return
	}

	err = gs.client.SendTransaction(context.Background(), signedTx)
	if err != nil {
		log.WithError(err).Error("send tx failed")
		return
	}

	txLog := log.WithField("txHash", signedTx.Hash().Hex())
	txLog.Info("tx broadcasted")
	for i := 0; i < 5; i++ {
		time.Sleep(time.Second * 2)
		receipt, err := gs.client.TransactionReceipt(context.Background(), signedTx.Hash())
		if err != nil && err == ethereum.NotFound {
			txLog.Info("tx receipt not found, retrying...")
			continue
		}
		if err != nil {
			txLog.WithError(err).Error("send gas tx failed")
			return
		}

		if receipt.Status != 1 {
			txLog.Error("send gas tx failed")
			return
		}
		txLog.Info("send gas tx confirmed")
		return
	}
	txLog.Error("tx receipt not found, timeout")
}

func (gs *GasSender) login(task *gasTask) {
	log.WithFields(log.Fields{
		"address":   task.user.Hex(),
		"userId":    task.userId,
		"inviterId": task.inviterId,
	}).Info("new login task")

	opts, err := bind.NewKeyedTransactorWithChainID(gs.privateKey, gs.chainId)
	if err != nil {
		log.WithError(err).Error("create transactor failed")
		return
	}

	opts.GasPrice = big.NewInt(1000000000)
	opts.GasLimit = 2000000

	signedTx, err := gs.aonUser.LoginByServer(opts, task.user, task.userId, task.inviterId)
	if err != nil {
		log.WithError(err).Error("send tx failed")
		return
	}
	txLog := log.WithField("txHash", signedTx.Hash().Hex())
	txLog.Info("tx broadcasted")
	for i := 0; i < 5; i++ {
		time.Sleep(time.Second * 2)
		receipt, err := gs.client.TransactionReceipt(context.Background(), signedTx.Hash())
		if err != nil && err == ethereum.NotFound {
			txLog.Info("tx receipt not found, retrying...")
			continue
		}
		if err != nil {
			txLog.WithError(err).Error("send gas tx failed")
			return
		}

		if receipt.Status != 1 {
			txLog.Error("send gas tx failed")
			return
		}
		txLog.Info("send gas tx confirmed")
		return
	}
	txLog.Error("tx receipt not found, timeout")
}
