package dao

import (
	"context"
	"errors"
	"math/big"
	"time"

	"github.com/ethereum/go-ethereum"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/core/types"
	"github.com/ethereum/go-ethereum/crypto"
	log "github.com/sirupsen/logrus"
)

func (d *Dao) BroadcastTx(to common.Address, nonce uint64, value *big.Int, calldata []byte) (txHash common.Hash, send bool, err error) {
	// todo 未来改成并发可以传privatekey进来
	ecdsaKey := crypto.ToECDSAUnsafe(common.FromHex(d.c.Chain.SenderPrivateKey))

	tx := &types.LegacyTx{
		Nonce:    nonce,
		GasPrice: big.NewInt(1000000000),
		Gas:      2000000,
		To:       &to,
		Value:    value,
		Data:     calldata,
	}

	signer := types.NewEIP155Signer(d.chainId)
	signedTx, err := types.SignNewTx(ecdsaKey, signer, tx)
	if err != nil {
		log.WithError(err).Error("sign tx failed")
		return
	}

	err = d.ethClient.SendTransaction(context.Background(), signedTx)
	if err != nil {
		return
	}
	return signedTx.Hash(), true, nil
}

func (d *Dao) WaitForReceipt(txHash common.Hash, timeout ...time.Duration) (receipt *types.Receipt, err error) {
	ctx := context.Background()
	if len(timeout) > 0 {
		ctx, _ = context.WithTimeout(ctx, timeout[0])
	}

	queryTicker := time.NewTicker(time.Second)
	defer queryTicker.Stop()
	for {
		receipt, err := d.ethClient.TransactionReceipt(ctx, txHash)
		if err == nil {
			return receipt, nil
		}

		if !errors.Is(err, ethereum.NotFound) {
			return nil, err
		}

		// Wait for the next round.
		select {
		case <-ctx.Done():
			return nil, ctx.Err()
		case <-queryTicker.C:
		}
	}
}

func (d *Dao) GetNonce(address common.Address) (uint64, error) {
	return d.ethClient.NonceAt(context.Background(), address, nil)
}
