package service

import (
	"encoding/hex"
	"sdk_api/constant"
	"sdk_api/contract/aonUser"
	dbModel "sdk_api/model/db"
	"sdk_api/util"
	"strings"

	"github.com/ethereum/go-ethereum/common"
	"github.com/google/uuid"
	log "github.com/sirupsen/logrus"
)

func (s *Service) CheckUser(platform string, platformId string) (exist bool, uid, keystore string, err error) {
	user, err := s.d.CheckUser(platform, platformId)
	if err != nil {
		log.WithError(err).Error("get user failed")
		return
	}
	if user == nil {
		uid := uuid.New().String()
		err = s.d.CreateEmptyUser(&dbModel.User{
			Uid:        uid,
			Platform:   platform,
			PlatformId: platformId,
		})
		if err != nil {
			log.WithError(err).Error("create user failed")
			return false, "", "", err
		}

		return false, uid, "", nil
	}
	return user.Keystore != "", user.Uid, user.Keystore, nil
}

func (s *Service) SetKeystore(uid, address, keystore string) (ok bool, err error) {
	user, err := s.d.GetUserByUid(uid)
	if err != nil {
		log.WithError(err).Error("get user failed")
		return
	}
	if user.Keystore != "" {
		log.WithField("uid", uid).Warn("keystore already exist")
		return false, nil
	}
	err = s.d.SetKeystore(uid, address, keystore)
	if err != nil {
		log.WithError(err).Error("set keystore failed")
		return
	}
	return true, nil
}

func (s *Service) AONSendGas(address string) (err error) {
	sendGasTask := &dbModel.Task{
		TaskId:    util.GenUUID(),
		Action:    "sendGas",
		ToAddress: address,
		Value:     constant.Ether.String(),
	}
	err = s.d.CreateTask(sendGasTask)
	if err != nil {
		log.WithError(err).Error("create task failed")
		return
	}
	return

}

func (s *Service) AONServerLogin(address, userId, inviter string) (taskId string, err error) {
	abi, _ := aonUser.AonUserMetaData.GetAbi()
	calldata, err := abi.Pack("loginByServer", common.HexToAddress(address), userId, inviter)
	if err != nil {
		log.WithError(err).Error("pack calldata failed")
		log.WithFields(log.Fields{
			"address": address,
			"userId":  userId,
			"inviter": inviter,
			"error":   err.Error(),
		}).Error("pack calldata failed")
		return
	}

	serverLoginTask := &dbModel.Task{
		TaskId:    util.GenUUID(),
		Uid:       userId,
		ToAddress: strings.ToLower(s.cfg.Chain.UserContract),
		Action:    "serverLogin",
		Value:     constant.ZeroValue.String(),
		Calldata:  hex.EncodeToString(calldata),
	}

	err = s.d.CreateTask(serverLoginTask)
	if err != nil {
		log.WithError(err).Error("create task failed")
		return
	}
	return serverLoginTask.TaskId, nil
}
