package dao

import (
	"context"
	"errors"
	"fmt"
	"io"
	"net/http"
	"strings"
	"taskcenter/constant"
	dbModel "taskcenter/model/db"
	"time"

	log "github.com/sirupsen/logrus"
	"github.com/tidwall/gjson"
)

func (d *Dao) GetTGGroupInfo(chatId int) (title, username string, err error) {
	url := fmt.Sprintf("%s/api/v1/chat?chatId=%d", strings.TrimSuffix(d.c.TGTask.URL, "/"), chatId)
	log.WithField("url", url).Debug("get tg group info")
	data, err := httpGet(url)
	if err != nil {
		return
	}

	retCode := gjson.Get(string(data), "code").Int()
	if retCode == 1 {
		return "", "", errors.New(string(data))
	}

	return gjson.Get(string(data), "data.title").String(), gjson.Get(string(data), "data.username").String(), nil
}

func (d *Dao) CheckTGJoin(userId string, chatId int) (ok, premium bool, err error) {
	url := fmt.Sprintf("%s/api/v1/user/joined?chatId=%d&userId=%s", strings.TrimSuffix(d.c.TGTask.URL, "/"), chatId, userId)
	log.WithField("url", url).Debug("check tg join")
	data, err := httpGet(url)
	if err != nil {
		return false, false, err
	}

	retCode := gjson.Get(string(data), "code").Int()
	if retCode == 1 {
		return false, false, errors.New(string(data))
	}

	return gjson.Get(string(data), "data.joined").Bool(), gjson.Get(string(data), "data.isPremium").Bool(), nil
}

func (d *Dao) CheckTGActive(userId string, chatId int) (msgCount int, err error) {
	url := fmt.Sprintf("%s/api/v1/user/active?chatId=%d&userId=%s", strings.TrimSuffix(d.c.TGTask.URL, "/"), chatId, userId)
	log.WithField("url", url).Debug("check tg active")
	data, err := httpGet(url)
	if err != nil {
		return 0, err
	}

	retCode := gjson.Get(string(data), "code").Int()
	if retCode == 1 {
		return 0, errors.New(string(data))
	}

	return int(gjson.Get(string(data), "data.msgCount").Int()), nil
}

func httpGet(url string) ([]byte, error) {
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()
	req, err := http.NewRequestWithContext(ctx, "GET", url, nil)
	if err != nil {
		return nil, err
	}
	resp, err := http.DefaultClient.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	return io.ReadAll(resp.Body)
}

func httpPost(url string, body io.Reader) ([]byte, error) {
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()
	req, err := http.NewRequestWithContext(ctx, "POST", url, body)
	if err != nil {
		return nil, err
	}
	resp, err := http.DefaultClient.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	return io.ReadAll(resp.Body)
}

func (d *Dao) CheckTGBind(userId string) (ok bool, err error) {
	// 任务完成表中是否存在该用户
	sql := `SELECT COUNT(1) FROM %s AS T JOIN %s AS TH ON T.id = TH.task_id WHERE TH.user_id = ? AND T.platform = ? AND T.action = ? AND T.start >= ? AND T.end <= ? AND T.deleted_at IS NULL AND TH.deleted_at IS NULL`
	sql = fmt.Sprintf(sql, (&dbModel.Task{}).TableName(), (&dbModel.TaskHistory{}).TableName())
	var ct int64
	err = d.db.Raw(sql, userId, constant.TaskPlatformTelegram, constant.TaskActionBind, time.Now().Unix(), time.Now().Unix()).Count(&ct).Error
	return ct > 0, err
}
