package service

import (
	"fmt"
	"taskcenter/constant"
	"taskcenter/dao"
	apiModel "taskcenter/model/api"
	dbModel "taskcenter/model/db"
	"taskcenter/util"
	"time"

	log "github.com/sirupsen/logrus"
)

func (s *Service) CreateTask(req *apiModel.CreateTaskRequest) (taskId int, err error) {
	task := &dbModel.Task{
		Id:          util.GenFlakeID(),
		GroupId:     req.GroupId,
		Platform:    req.Platform,
		Action:      req.Action,
		Description: req.Description,
		Reward:      req.Reward,
		Start:       req.Start,
		End:         req.End,
		Daily:       *req.Daily,
		Enable:      true,
		Pass:        req.Pass,
	}
	task.TwitterAPIKey = req.TwitterAPIKey
	task.TwitterAPISecret = req.TwitterAPISecret
	task.TwitterAccessToken = req.TwitterAccessToken
	task.TwitterAccessSecret = req.TwitterAccessSecret
	task.TwitterToken = req.TwitterToken

	config := dao.TwitterConfig{
		APIKey:            req.TwitterAPIKey,
		APISecret:         req.TwitterAPISecret,
		AccessToken:       req.TwitterAccessToken,
		AccessTokenSecret: req.TwitterAccessSecret,
		TwitterToken:      req.TwitterToken,
	}
	if req.Platform == constant.TaskPlatformTwitter {
		// 获取推特信息
		userId, userHandle, _, err := s.d.GetTwitterInfo(config)
		if err != nil {
			log.WithError(err).Error("get twitter info error")
			return 0, err
		}
		task.TwitterHandle = userHandle
		task.TwitterUserId = userId
		var twitterAPIAction string
		switch req.Action {
		case constant.TaskActionFollow:
			twitterAPIAction = constant.TwitterAPIActionFollow
			task.Url = fmt.Sprintf("https://x.com/intent/follow?screen_name=%s", userHandle)
		case constant.TaskActionLike:
			twitterAPIAction = constant.TwitterAPIActionLike
			task.TweetId = req.TweetId
			task.Url = fmt.Sprintf("https://x.com/intent/like?tweet_id=%d", req.TweetId)
		case constant.TaskActionRetweet:
			twitterAPIAction = constant.TwitterAPIActionRetweet
			task.TweetId = req.TweetId
			task.Url = fmt.Sprintf("https://x.com/intent/retweet?tweet_id=%d", req.TweetId)
		}
		var twitterHandle string
		// 关注时传handle
		if req.Action == constant.TaskActionFollow {
			twitterHandle = task.TwitterHandle
		}
		// 推特任务中心，启动任务
		err = s.d.DoTweetTask(task.TwitterUserId, task.TweetId, twitterHandle, twitterAPIAction, true, config)
		if err != nil {
			log.WithError(err).Error("do tweet task error")
			return 0, err
		}

	}
	if req.Platform == constant.TaskPlatformTelegram {
		_, username, err := s.d.GetTGGroupInfo(req.TelegramChatId)
		if err != nil {
			log.WithError(err).Error("get telegram group info error")
			return 0, err
		}

		task.TelegramChatId = req.TelegramChatId
		task.TelegramChatUsername = username
		task.Url = fmt.Sprintf("https://t.me/%s", username)
		if req.Action == constant.TaskActionActive {
			task.TelegramActiveThreshold = req.TelegramActiveThreshold
		}
	}

	err = s.d.CreateTask(task)
	if err != nil {
		log.WithError(err).Error("create group task error")
		return 0, err
	}
	return task.Id, nil
}

func (s *Service) EditTask(req *apiModel.EditTaskRequest) (err error) {
	err = s.d.EditTask(req.TaskId, req.Delete, req.Enable)
	if err != nil {
		log.WithError(err).Error("edit task error")
	}
	return
}

func (s *Service) GetTaskOwner(taskId, groupId int) (userId string, err error) {
	userId, err = s.d.GetTaskOwner(taskId, groupId)
	if err != nil {
		log.WithError(err).Error("get task owner error")
	}
	return
}

func (s *Service) CheckTask(taskId int, userId string) (exist, done bool, expired bool, err error) {
	exist = true
	task, err := s.d.GetTaskDetail(taskId)
	if err != nil {
		log.WithError(err).Error("get task error")
		return
	}
	if task == nil {
		exist = false
		return
	}
	log.Debugf("task: %+v", task)

	// 超出时间后只查询不再更新状态
	if int64(task.Start) > time.Now().Unix() || int64(task.End) < time.Now().Unix() {
		expired = true
	}

	done, err = s.d.IsTaskDone(taskId, userId)
	if err != nil {
		log.WithError(err).Error("is task done error")
		return
	}

	return
}

func (s *Service) GetTaskDetail(taskId int) (task *dbModel.Task, err error) {
	task, err = s.d.GetTaskDetail(taskId)
	if err != nil {
		log.WithError(err).Error("get task error")
	}
	return
}

func (s *Service) GetTaskResult(isDailyTask bool, taskId int, userId string) (status string, submittedAt time.Time, err error) {
	status, submittedAt, err = s.d.GetTaskResult(isDailyTask, taskId, userId)
	if err != nil {
		log.WithError(err).Error("get task result error")
	}
	return
}

func (s *Service) SubmitTask(taskId int, userId string, passTask, isDailyTask bool, status string) (exist bool, err error) {
	if passTask {
		status = constant.TaskHistoryStatusSuccess
	}

	exist, err = s.d.CreateTaskHistory(taskId, userId, isDailyTask, status)
	if err != nil {
		log.WithError(err).Error("create task history error")
	}
	return
}
