package service

import (
	"fmt"
	"sdk_api/constant"
	apiModel "sdk_api/model/api"
	dbModel "sdk_api/model/db"
	"sdk_api/util"

	log "github.com/sirupsen/logrus"
)

func (s *Service) CreateGroup(req *apiModel.CreateGroupRequest) (gid int, err error) {
	g := &dbModel.TaskGroup{
		Id:          util.GenFlakeID(),
		ProjectId:   req.ProjectId,
		Description: req.Description,
	}
	err = s.d.CreateGroup(g)
	if err != nil {
		log.WithError(err).Error("create group error")
		return
	}

	project, err := s.d.GetProject(req.ProjectId)
	if err != nil {
		log.WithError(err).Error("get project error")
		return 0, err
	}

	for _, task := range req.Tasks {
		daily := 0
		if task.Daily {
			daily = 1
		}
		gt := &dbModel.Task{
			Id:            util.GenFlakeID(),
			GroupId:       g.Id,
			Platform:      task.Platform,
			Action:        task.Action,
			TwitterHandle: project.TwitterHandle,
			TwitterUserId: project.TwitterUserId,
			Description:   task.Description,
			Reward:        task.Reward,
			Start:         task.Start,
			End:           task.End,
			Daily:         daily,
		}
		if task.Platform == constant.TaskPlatformTwitter {
			var twitterAPIAction string
			switch task.Action {
			case constant.TaskActionFollow:
				twitterAPIAction = constant.TwitterAPIActionFollow
				gt.Url = fmt.Sprintf("https://x.com/intent/follow?screen_name=%s", project.TwitterHandle)
			case constant.TaskActionLike:
				twitterAPIAction = constant.TwitterAPIActionLike
				gt.TweetId = task.TweetId
				gt.Url = fmt.Sprintf("https://x.com/intent/like?tweet_id=%d", task.TweetId)
			case constant.TaskActionRetweet:
				twitterAPIAction = constant.TwitterAPIActionRetwitter
				gt.TweetId = task.TweetId
				gt.Url = fmt.Sprintf("https://x.com/intent/retweet?tweet_id=%d", task.TweetId)
			}
			tweetId := gt.TweetId
			if task.Action == constant.TaskActionFollow {
				tweetId = gt.TwitterUserId
			}
			// 推特任务中心，启动任务
			err = s.d.DoTweetTask(gt.TwitterUserId, tweetId, twitterAPIAction, true)
			if err != nil {
				log.WithError(err).Error("do tweet task error")
				return 0, err
			}

		}
		if task.Platform == constant.TaskPlatformTelegram {
			gt.TelegramChatId = project.TelegramChatId
			gt.TelegramChatUsername = project.TelegramChatUsername
			gt.Url = fmt.Sprintf("https://t.me/%s", project.TelegramChatUsername)
		}

		err = s.d.CreateGroupTask(gt)
		if err != nil {
			log.WithError(err).Error("create group task error")
			return 0, err
		}
	}
	return g.Id, nil
}

func (s *Service) GetGroup(gid int) (resp *apiModel.GetGroupResponse, err error) {
	resp = &apiModel.GetGroupResponse{Tasks: make([]apiModel.Task, 0)}
	g, err := s.d.GetGroup(gid)
	if err != nil {
		log.WithError(err).Error("get group error")
		return
	}
	if g == nil {
		log.WithError(err).Error("group not found")
		return nil, nil
	}

	resp.ProjectId = g.ProjectId
	resp.Description = g.Description

	tasks, err := s.d.GetGroupTasks(g.Id)
	if err != nil {
		log.WithError(err).Error("get group tasks error")
		return
	}

	for _, task := range tasks {
		resp.Tasks = append(resp.Tasks, apiModel.Task{
			TaskId:      task.Id,
			Platform:    task.Platform,
			Action:      task.Action,
			Url:         task.Url,
			Description: task.Description,
			Reward:      task.Reward,
			Start:       task.Start,
			End:         task.End,
			Daily:       task.Daily == 1,
			TweetId:     task.TweetId,
		})
	}
	return
}

func (s *Service) GetGroupList(page, pageSize int) (resp *apiModel.GetGroupListResponse, err error) {
	resp = &apiModel.GetGroupListResponse{}
	list, count, err := s.d.GetGroupList(page, pageSize)
	if err != nil {
		log.WithError(err).Error("get group list error")
		return
	}
	resp.TotalCount = count
	resp.Items = make([]*apiModel.GetGroupResponse, 0)
	for _, v := range list {
		resp.Items = append(resp.Items, &apiModel.GetGroupResponse{
			GroupId:     v.Id,
			ProjectId:   v.ProjectId,
			Description: v.Description,
		})

		tasks, err := s.d.GetGroupTasks(v.Id)
		if err != nil {
			log.WithError(err).Error("get group tasks error")
			return resp, err
		}

		log.Error("len: ", len(tasks))

		for _, task := range tasks {
			resp.Items[len(resp.Items)-1].Tasks = append(resp.Items[len(resp.Items)-1].Tasks, apiModel.Task{
				TaskId:      task.Id,
				Platform:    task.Platform,
				Action:      task.Action,
				Url:         task.Url,
				Description: task.Description,
				Reward:      task.Reward,
				Start:       task.Start,
				End:         task.End,
				Daily:       task.Daily == 1,
				TweetId:     task.TweetId,
			})
		}
	}
	return
}
