package service

import (
	"fmt"
	"sdk_api/constant"
	"time"

	log "github.com/sirupsen/logrus"
)

func (s *Service) CheckTask(taskId int, userId string) (exist, done bool, expired bool, err error) {
	exist = true
	task, err := s.d.GetTaskDetail(taskId)
	if err != nil {
		log.WithError(err).Error("get task error")
		return
	}
	if task == nil {
		exist = false
		return
	}
	log.Debugf("task: %+v", task)

	// 超出时间后只查询不再更新状态
	if int64(task.Start) > time.Now().Unix() || int64(task.End) < time.Now().Unix() {
		expired = true
	}

	done, err = s.d.IsTaskDone(taskId, userId)
	if err != nil {
		log.WithError(err).Error("is task done error")
		return
	}

	return
}

func (s *Service) SyncTask(taskId int, userId string) (done bool, err error) {
	task, err := s.d.GetTaskDetail(taskId)
	if err != nil {
		log.WithError(err).Error("get task error")
		return
	}
	if task == nil {
		return
	}

	switch task.Platform {
	case constant.TaskPlatformTelegram:
		telegramUserId, err := s.d.GetProviderId(userId, constant.TaskPlatformTelegram)
		if err != nil {
			log.WithError(err).Error("get provider telegram user id error")
			return false, err
		}

		if telegramUserId == "" {
			return false, nil
		}

		switch task.Action {
		case constant.TaskActionJoin, constant.TaskActionActive:
			var taskDone bool
			if task.Action == constant.TaskActionJoin {
				taskDone, err = s.d.CheckTGJoin(telegramUserId, task.TelegramChatId)
			} else {
				taskDone, err = s.d.CheckTGActive(telegramUserId, task.TelegramChatId)
			}

			if err != nil {
				log.WithError(err).Errorf("check tg %s error", task.Action)
				return false, err
			}
			if !taskDone {
				return false, nil
			}
			err = s.d.CreateTaskHistory(taskId, userId, task.Daily == 1)
			if err != nil {
				log.WithError(err).Error("create task history error")
				return false, err
			}
			return true, nil
		default:
			return false, fmt.Errorf("unknown task action: %s", task.Action)
		}
	case constant.TaskPlatformTwitter:
		twitterUserId, err := s.d.GetProviderId(userId, constant.TaskPlatformTwitter)
		if err != nil {
			log.WithError(err).Error("get provider twitter user id error")
			return false, err
		}

		switch task.Action {
		case constant.TaskActionFollow:
			followed, err := s.d.CheckTwitterFollow(task.TwitterUserId, twitterUserId)
			if err != nil {
				log.WithError(err).Error("check twitter follow error")
				return false, err
			}
			if !followed {
				return false, nil
			}
			err = s.d.CreateTaskHistory(taskId, userId, task.Daily == 1)
			if err != nil {
				log.WithError(err).Error("create task history error")
				return false, err
			}
			return true, nil
		case constant.TaskActionLike, constant.TaskActionRetweet, constant.TaskActionReply:
			var taskDone bool
			if task.Action == constant.TaskActionLike {
				taskDone, err = s.d.CheckTwitterLike(task.TweetId, twitterUserId, 0, 0)
			} else {
				taskDone, err = s.d.CheckTwitterRetweet(task.TweetId, twitterUserId, 0, 0)
			}
			if err != nil {
				log.WithError(err).Errorf("check twitter %s error", task.Action)
				return false, err
			}
			if !taskDone {
				return false, nil
			}
			err = s.d.CreateTaskHistory(taskId, userId, task.Daily == 1)
			if err != nil {
				log.WithError(err).Error("create task history error")
				return false, err
			}
			return true, nil

		default:
			return false, fmt.Errorf("unknown task action: %s", task.Action)
		}
	default:
		return false, fmt.Errorf("unknown task platform: %s", task.Platform)
	}
}
