package sync

import (
	"fmt"
	"taskcenter/constant"
	"time"

	log "github.com/sirupsen/logrus"
)

func (s *Sync) ProcessTasks() {
	ticker := time.NewTicker(time.Second)
	defer ticker.Stop()
	for {
		select {
		case <-ticker.C:
			tasks, err := s.d.GetUnprocessedTasks()
			if err != nil {
				log.WithError(err).Error("get unprocessed tasks error")
				return
			}

			for _, task := range tasks {
				status := constant.TaskHistoryStatusSuccess
				ok, err := s.SyncTask(task.Id, task.UserId)
				if err != nil {
					log.WithError(err).Error("sync task error")
				}
				if !ok {
					status = constant.TaskHistoryStatusRetry
				}
				log.WithFields(log.Fields{"task_id": task.Id, "error": err, "status": status}).Info("sync task")
				err = s.d.UpdateTaskHistory(task.Id, status)
				if err != nil {
					log.WithError(err).Error("update task history error")
				}
			}
		}
	}

}

func (s *Sync) SyncTask(taskId int, userId string) (ok bool, err error) {
	task, err := s.d.GetTaskDetail(taskId)
	if err != nil {
		log.WithError(err).Error("get task error")
		return
	}
	if task == nil {
		return
	}

	switch task.Platform {
	case constant.TaskPlatformTelegram:
		telegramUserId, err := s.d.GetProviderId(userId, constant.TaskPlatformTelegram)
		if err != nil {
			log.WithError(err).Error("get provider telegram user id error")
			return false, err
		}

		if telegramUserId == "" {
			return false, nil
		}

		switch task.Action {
		case constant.TaskActionJoin, constant.TaskActionActive:
			var taskDone bool
			if task.Action == constant.TaskActionJoin {
				taskDone, err = s.d.CheckTGJoin(telegramUserId, task.TelegramChatId)
			} else {
				taskDone, err = s.d.CheckTGActive(telegramUserId, task.TelegramChatId)
			}

			if err != nil {
				log.WithError(err).Errorf("check tg %s error", task.Action)
				return false, err
			}
			return taskDone, nil
		default:
			return false, fmt.Errorf("unknown task action: %s", task.Action)
		}
	case constant.TaskPlatformTwitter:
		twitterUserId, err := s.d.GetProviderId(userId, constant.TaskPlatformTwitter)
		if err != nil {
			log.WithError(err).Error("get provider twitter user id error")
			return false, err
		}

		switch task.Action {
		case constant.TaskActionFollow:
			followed, err := s.d.CheckTwitterFollow(task.TwitterHandle, twitterUserId)
			if err != nil {
				log.WithError(err).Error("check twitter follow error")
				return false, err
			}
			if !followed {
				return false, nil
			}
			return true, nil
		case constant.TaskActionLike, constant.TaskActionRetweet, constant.TaskActionReply:
			var taskDone bool
			if task.Action == constant.TaskActionLike {
				taskDone, err = s.d.CheckTwitterLike(task.TweetId, twitterUserId, 0, 0)
			} else {
				taskDone, err = s.d.CheckTwitterRetweet(task.TweetId, twitterUserId, 0, 0)
			}
			if err != nil {
				log.WithError(err).Errorf("check twitter %s error", task.Action)
				return false, err
			}
			return taskDone, nil
		default:
			return false, fmt.Errorf("unknown task action: %s", task.Action)
		}
	default:
		return false, fmt.Errorf("unknown task platform: %s", task.Platform)
	}
}
