package server

import (
	"net/url"
	"strconv"
	"taskcenter/constant"
	apiModel "taskcenter/model/api"
	"taskcenter/util"

	"github.com/gin-gonic/gin"
	log "github.com/sirupsen/logrus"
)

func createTask(c *gin.Context) {
	req := &apiModel.CreateTaskRequest{}
	if err := c.ShouldBindJSON(req); err != nil {
		c.JSON(200, withError(constant.InvalidParam))
		return
	}
	req.UserId = c.GetString("userId")

	ok, err := srv.CheckGroup(req.GroupId, req.UserId)
	if err != nil {
		c.JSON(200, withError(constant.InternalError))
		return
	}
	if !ok {
		c.JSON(200, withError(constant.InvalidParam))
		return
	}

	taskId, err := srv.CreateTask(req)
	if err != nil {
		c.JSON(200, withError(constant.InternalError))
		return
	}
	c.JSON(200, withSuccess(gin.H{"taskId": taskId}))
}

func editTask(c *gin.Context) {
	req := &apiModel.EditTaskRequest{}
	if err := c.ShouldBindJSON(req); err != nil {
		c.JSON(200, withError(constant.InvalidParam))
		return
	}
	req.UserId = c.GetString("userId")

	userId, err := srv.GetTaskOwner(req.TaskId, req.GroupId)
	if err != nil {
		c.JSON(200, withError(constant.InternalError))
		return
	}

	if userId != req.UserId {
		c.JSON(200, withError("permission denied"))
		return
	}

	if req.Delete {
		req.Enable = false
	}

	err = srv.EditTask(req)
	if err != nil {
		c.JSON(200, withError(constant.InternalError))
		return
	}
	c.JSON(200, withSuccess(""))
}

func checkTask(c *gin.Context) {
	_taskId := c.Param("tid")
	taskId, _ := strconv.Atoi(_taskId)
	// userId := c.Query("userId")
	userId := c.GetString("userId")

	if taskId == 0 || userId == "" {
		c.JSON(200, withError(constant.InvalidParam))
		return
	}

	task, err := srv.GetTaskDetail(taskId)
	if err != nil {
		c.JSON(200, withError(constant.InternalError))
		return
	}

	if task == nil {
		c.JSON(200, withError("task not found"))
		return
	}

	status, _, err := srv.GetTaskResult(task.Daily, taskId, userId)
	if err != nil {
		c.JSON(200, withError(constant.InternalError))
		return
	}

	if status == constant.TaskHistoryStatusRetry {
		c.JSON(200, withSuccess(gin.H{"status": status, "msg": "The task is not completed, please try again."}))
		return
	}

	c.JSON(200, withSuccess(gin.H{"status": status}))
}

// submitTask 提交任务，等待后台检查
func submitTask(c *gin.Context) {
	_taskId := c.Param("tid")
	taskId, _ := strconv.Atoi(_taskId)
	// userId := c.Query("userId")
	userId := c.GetString("userId")
	initData := c.Query("initData")
	initData, _ = url.QueryUnescape(initData)
	if taskId == 0 || userId == "" {
		c.JSON(200, withError(constant.InvalidParam))
		return
	}

	task, err := srv.GetTaskDetail(taskId)
	if err != nil {
		c.JSON(200, withError(constant.InternalError))
		return
	}

	if task == nil {
		c.JSON(200, withError("task not found"))
		return
	}
	status, _, err := srv.GetTaskResult(task.Daily, taskId, userId)
	if err != nil {
		c.JSON(200, withError(constant.InternalError))
		return
	}

	if status == constant.TaskHistoryStatusPending {
		c.JSON(200, withError("task already submitted"))
		return
	}

	if status == constant.TaskHistoryStatusSuccess {
		c.JSON(200, withError("task already done"))
		return
	}

	// 有initdata必须为appbase premiunsignin
	if task.Action == constant.TaskActionPremiumSignIn && initData == "" {
		c.JSON(200, withError(constant.InvalidParam))
		return
	}

	// 没有initdata，sync模块处理状态
	if initData == "" {
		_, err = srv.SubmitTask(taskId, userId, task.Daily, constant.TaskHistoryStatusPending)
		if err != nil {
			c.JSON(200, withError(constant.InternalError))
			return
		}
		c.JSON(200, withSuccess(gin.H{}))
		return
	}

	// appbase tg会员签到, 直接在此处确定任务完成状态
	ok, isPremium, _, tgUserId := util.VerifyInitData(initData, conf.TGTask.BOTToken)
	telegramUserId, err := d.GetProviderId(userId, constant.TaskPlatformTelegram)
	if err != nil {
		log.WithError(err).Error("get provider telegram user id error")
		c.JSON(200, withError(constant.InternalError))
		return
	}

	if !ok {
		c.JSON(200, withError("invalid initData"))
		return
	}

	if tgUserId != telegramUserId {
		log.WithField(tgUserId, telegramUserId).Error("tgUserId != telegramUserId")
		c.JSON(200, withError("invalid initData"))
		return
	}
	taskStatus := constant.TaskHistoryStatusRetry
	if isPremium {
		taskStatus = constant.TaskHistoryStatusSuccess
	}
	_, err = srv.SubmitTask(taskId, userId, task.Daily, taskStatus)
	if err != nil {
		c.JSON(200, withError(constant.InternalError))
		return
	}
	c.JSON(200, withSuccess(gin.H{}))
	return

}
