package dao

import (
	"fmt"
	"taskcenter/constant"
	dbModel "taskcenter/model/db"
	"taskcenter/util"
	"time"

	"gorm.io/gorm"
	"gorm.io/gorm/clause"
	"gorm.io/gorm/logger"
)

func (d *Dao) InitTaskAction() (err error) {
	for platform, actions := range constant.TaskAction {
		for _, action := range actions {
			t := &dbModel.TaskAction{
				Id:       util.GenFlakeID(),
				Platform: platform,
				Action:   action,
			}
			err = d.db.Clauses(clause.OnConflict{DoNothing: true}).Create(t).Error
			if err != nil {
				return err
			}
		}
	}
	return err
}

func (d *Dao) CreateProject(p *dbModel.Project) (err error) {
	return d.db.Create(p).Error
}

func (d *Dao) GetProjectList(page, pageSize int) (list []*dbModel.Project, totalCount int, err error) {
	var tmpCount int64
	err = d.db.Model(&dbModel.Project{}).Count(&tmpCount).Error
	if err != nil {
		return
	}
	return list, int(tmpCount), d.db.Limit(pageSize).Offset((page - 1) * pageSize).Find(&list).Error
}

func (d *Dao) GetProject(id int) (p *dbModel.Project, err error) {
	p = &dbModel.Project{}
	err = d.db.Where("id = ?", id).First(&p).Error
	if err == gorm.ErrRecordNotFound {
		return nil, nil
	}
	return p, err
}

func (d *Dao) CreateGroup(g *dbModel.TaskGroup) (err error) {
	return d.db.Create(g).Error
}

func (d *Dao) GetGroupList(page, pageSize int) (list []*dbModel.TaskGroup, totalCount int, err error) {
	var tmpCount int64
	err = d.db.Model(&dbModel.TaskGroup{}).Count(&tmpCount).Error
	if err != nil {
		return
	}
	totalCount = int(tmpCount)
	return list, totalCount, d.db.Limit(pageSize).Offset((page - 1) * pageSize).Find(&list).Error
	// 性能优化
	// sql := fmt.Sprintf("SELECT * FROM %[1]s t1 JOIN (SELECT id FROM %[1]s LIMIT ?, ?) t2 ON t1.id = t2.id", (&dbModel.Task{}).TableName())
	// return list, totalCount, d.db.Raw(sql, (page-1)*pageSize, pageSize).Scan(&list).Error
}

func (d *Dao) GetGroup(id int) (g *dbModel.TaskGroup, err error) {
	g = &dbModel.TaskGroup{}
	err = d.db.Where("id = ?", id).First(&g).Error
	if err == gorm.ErrRecordNotFound {
		return nil, nil
	}
	return g, err
}

func (d *Dao) GetGroupTasks(gid int, admin bool) (list []*dbModel.Task, err error) {
	if admin {
		return list, d.db.Where("group_id = ?", gid).Order("created_at ASC").Find(&list).Error
	}
	return list, d.db.Where("group_id = ? and enable = ?", gid, true).Order("created_at ASC").Find(&list).Error
}

func (d *Dao) CreateTask(gt *dbModel.Task) (err error) {
	return d.db.Create(gt).Error
}

func (d *Dao) GetTaskDetail(tid int) (t *dbModel.Task, err error) {
	t = &dbModel.Task{}
	err = d.db.Where("id = ?", tid).First(&t).Error
	if err == gorm.ErrRecordNotFound {
		return nil, nil
	}
	return t, err

}

func (d *Dao) GetTaskOwner(tid, gid int) (userId string, err error) {
	// 联表group task，判断是否是该用户创建的任务组
	sql := fmt.Sprintf(
		"SELECT t2.user_id FROM %s t1 LEFT JOIN %s t2 ON t1.group_id = t2.id WHERE t1.id = ? AND t2.id = ?",
		(&dbModel.Task{}).TableName(), (&dbModel.TaskGroup{}).TableName(),
	)
	err = d.db.Raw(sql, tid, gid).Scan(&userId).Error
	return
}

func (d *Dao) IsDailyTask(tid int) (ok bool, err error) {
	var count int64
	err = d.db.Model(&dbModel.Task{}).Where("id = ? and daily = ?", tid, true).Count(&count).Error
	if err != nil {
		return false, err
	}
	return count > 0, nil
}

// IsTaskDone 是否任务已完成
func (d *Dao) IsTaskDone(tid int, userId string) (ok bool, err error) {
	isDailyTask, err := d.IsDailyTask(tid)
	if err != nil {
		return false, err
	}

	var count int64
	if !isDailyTask {
		err = d.db.Model(&dbModel.TaskHistory{}).Where("task_id = ? and user_id = ?", tid, userId).Count(&count).Error
		return count > 0, err
	}

	err = d.db.Model(&dbModel.TaskHistory{}).
		Where("task_id = ? and user_id = ? and created_at >= ?", tid, userId, time.Now().UTC().Truncate(24*time.Hour)).
		Count(&count).Error
	return count > 0, err
}

func (d *Dao) GetTaskResult(isDailyTask bool, tid int, userId string) (status string, submittedAt time.Time, err error) {
	temp := &dbModel.TaskHistory{}
	tx := d.db.Session(&gorm.Session{Logger: logger.Default.LogMode(logger.Silent)})
	if !isDailyTask {
		err = tx.Model(&dbModel.TaskHistory{}).Where("task_id = ? and user_id = ?", tid, userId).First(temp).Error
		if err == gorm.ErrRecordNotFound {
			return constant.TaskHistoryStatusTodo, time.Time{}, nil
		}
		return temp.Status, temp.SubmittedAt, err
	}

	err = tx.Model(&dbModel.TaskHistory{}).
		Where("task_id = ? and user_id = ? and created_at >= ?", tid, userId, time.Now().UTC().Truncate(24*time.Hour)).
		First(temp).Error
	if err == gorm.ErrRecordNotFound {
		return constant.TaskHistoryStatusTodo, time.Time{}, nil
	}
	return temp.Status, temp.SubmittedAt, err
}

func (d *Dao) CreateTaskHistory(taskId int, userId string, isDailyTask bool, initStatus string) (exist bool, err error) {
	// 使用行锁，防止并发
	tx := d.db.Begin()
	defer func() {
		if err != nil {
			tx.Rollback()
		} else {
			tx.Commit()
		}
	}()
	temp := make([]*dbModel.TaskHistory, 0)
	if !isDailyTask {
		err = tx.Set("gorm:query_option",
			"FOR UPDATE").Where("task_id = ? and user_id = ?", taskId, userId).Take(&temp).Error
		if err == gorm.ErrRecordNotFound {
			err = tx.Create(&dbModel.TaskHistory{
				Id:         util.GenFlakeID(),
				TaskId:     taskId,
				UserId:     userId,
				Status:     initStatus,
				SubmittedAt: time.Now(),
			}).Error
			if err != nil {
				return false, err
			}
		}
		// 已存在，更新updated_at字段
		if err == nil {
			return true, tx.Model(&dbModel.TaskHistory{}).
				Where("task_id = ? and user_id = ?", taskId, userId).
				Limit(1).
				Updates(map[string]interface{}{"updated_at": time.Now(), "submitted_at": time.Now(), "status": initStatus}).Error
		}

		return err == nil, err
	}

	err = tx.Set("gorm:query_option", "FOR UPDATE").Where("task_id = ? and user_id = ? and created_at >= ?", taskId, userId, time.Now().UTC().Truncate(24*time.Hour)).Take(&temp).Error
	if err == gorm.ErrRecordNotFound {
		err = tx.Create(&dbModel.TaskHistory{
			Id:         util.GenFlakeID(),
			TaskId:     taskId,
			UserId:     userId,
			Status:     initStatus,
			SubmittedAt: time.Now(),
		}).Error
		if err != nil {
			return false, err
		}
	}
	// 已存在，更新updated_at字段
	if err == nil {
		return true, tx.Model(&dbModel.TaskHistory{}).
			Where("task_id = ? and user_id = ? and created_at >= ?", taskId, userId, time.Now().UTC().Truncate(24*time.Hour)).
			Limit(1).
			Updates(map[string]interface{}{"updated_at": time.Now(), "submitted_at": time.Now(), "status": initStatus}).
			Error
	}
	return err == nil, err
}

func (d *Dao) UpdateTaskHistory(id int, status string) (err error) {
	err = d.db.Model(&dbModel.TaskHistory{}).Where("id = ?", id).Update("status", status).Error
	return
}

func (d *Dao) GetUnprocessedTasks() (tasks []*dbModel.TaskHistory, err error) {
	// twitter延迟3分钟，tg延迟1分钟，appbase不延迟
	sql := fmt.Sprintf(
		`SELECT TH.* FROM %s AS TH JOIN %s AS T ON TH.task_id = T.id WHERE TH.status = ? AND ((T.platform = ? AND TH.updated_at <= NOW() - INTERVAL '3 minutes') OR (T.platform = ? AND TH.updated_at <= NOW() - INTERVAL '1 minutes') OR (T.platform = ?)  )`,
		(&dbModel.TaskHistory{}).TableName(), (&dbModel.Task{}).TableName(),
	)
	// ignore log
	tx := d.db.Session(&gorm.Session{Logger: logger.Default.LogMode(logger.Error)})
	err = tx.Raw(sql, constant.TaskHistoryStatusPending, constant.TaskPlatformTwitter, constant.TaskPlatformTelegram, constant.TaskPlatformAppbase).Scan(&tasks).Error
	return
}

func (d *Dao) IsAdminUser(userId string) (ok bool, err error) {
	var ct int64
	err = d.db.Model(&dbModel.AdminUser{}).Where("user_id = ?", userId).Count(&ct).Error
	return ct > 0, err
}

func (d *Dao) GetUnstoppedTasks() {

}

func (d *Dao) EditTask(taskId int, delete bool, enable bool) (err error) {
	if delete {
		err = d.db.Model(&dbModel.Task{}).Where("id = ?", taskId).Updates(map[string]interface{}{
			"enable":     false,
			"updated_at": "now()",
			"deleted_at": "now()",
		}).Error
	} else {
		err = d.db.Model(&dbModel.Task{}).Where("id = ?", taskId).Updates(map[string]interface{}{
			"enable":     enable,
			"updated_at": "now()",
		}).Error
	}

	return
}
