package service

import (
	"sync"
	"taskcenter/constant"
	apiModel "taskcenter/model/api"
	dbModel "taskcenter/model/db"
	"taskcenter/util"

	log "github.com/sirupsen/logrus"
)

func (s *Service) CreateGroup(req *apiModel.CreateGroupRequest) (gid int, err error) {
	g := &dbModel.TaskGroup{
		Id:          util.GenFlakeID(),
		UserId:      req.UserId,
		Description: req.Description,
	}
	err = s.d.CreateGroup(g)
	if err != nil {
		log.WithError(err).Error("create group error")
		return
	}
	return g.Id, nil
}

func (s *Service) CheckGroup(gid int, userId string) (exist bool, err error) {
	g, err := s.d.GetGroup(gid)
	if err != nil {
		log.WithError(err).Error("get group error")
		return
	}

	if g == nil {
		return false, nil
	}
	if g.UserId != userId {
		return false, nil
	}
	return true, nil
}

func (s *Service) GetGroup(gid int, userId string, admin bool) (resp *apiModel.GetGroupResponse, err error) {
	resp = &apiModel.GetGroupResponse{Tasks: make([]apiModel.Task, 0)}
	g, err := s.d.GetGroup(gid)
	if err != nil {
		log.WithError(err).Error("get group error")
		return
	}
	if g == nil {
		log.WithError(err).Error("group not found")
		return nil, nil
	}

	resp.Description = g.Description

	tasks, err := s.d.GetGroupTasks(g.Id, admin)
	if err != nil {
		log.WithError(err).Error("get group tasks error")
		return
	}

	taskResults := make([]apiModel.Task, len(tasks))
	var wg sync.WaitGroup
	var errCh = make(chan error, len(tasks))

	for i := range tasks {
		wg.Add(1)
		go func(idx int) {
			defer wg.Done()
			task := tasks[idx]
			_task := apiModel.Task{
				TaskId:      task.Id,
				Platform:    task.Platform,
				Action:      task.Action,
				Url:         task.Url,
				Description: task.Description,
				Reward:      task.Reward,
				Start:       task.Start,
				End:         task.End,
				Daily:       task.Daily,
				TweetId:     task.TweetId,
				Enable:      task.Enable,
				Pass:        task.Pass,
			}

			status, submittedAt, err := s.GetTaskResult(_task.Daily, _task.TaskId.(int), userId)
			if err != nil {
				log.WithError(err).Error("group check task")
				errCh <- err
				return
			}
			_task.Status = status
			if status == constant.TaskHistoryStatusRetry {
				_task.Msg = "The task is not completed, please try again."
			}
			_task.SubmitTimestamp = int(submittedAt.Unix())
			taskResults[idx] = _task
		}(i)
	}

	wg.Wait()
	if len(errCh) > 0 {
		err = <-errCh
		close(errCh)
		return
	}

	// 根据tasks中的created_at排序，tasks[i].id == taskResults[i].TaskId

	rets := make([]apiModel.Task, len(tasks))
	for i := range tasks {
		for j := range taskResults {
			if tasks[i].Id == taskResults[j].TaskId {
				rets[i] = taskResults[j]
				break
			}
		}
	}
	resp.Tasks = rets
	return
}

func (s *Service) GetGroupList(page, pageSize int, admin bool) (resp *apiModel.GetGroupListResponse, err error) {
	resp = &apiModel.GetGroupListResponse{}
	list, count, err := s.d.GetGroupList(page, pageSize)
	if err != nil {
		log.WithError(err).Error("get group list error")
		return
	}
	resp.TotalCount = count
	resp.Items = make([]*apiModel.GetGroupResponse, 0)
	for _, v := range list {
		resp.Items = append(resp.Items, &apiModel.GetGroupResponse{
			GroupId:     v.Id,
			Description: v.Description,
		})

		tasks, err := s.d.GetGroupTasks(v.Id, admin)
		if err != nil {
			log.WithError(err).Error("get group tasks error")
			return resp, err
		}

		for _, task := range tasks {
			resp.Items[len(resp.Items)-1].Tasks = append(resp.Items[len(resp.Items)-1].Tasks, apiModel.Task{
				TaskId:      task.Id,
				Platform:    task.Platform,
				Action:      task.Action,
				Url:         task.Url,
				Description: task.Description,
				Reward:      task.Reward,
				Start:       task.Start,
				End:         task.End,
				Daily:       task.Daily,
				TweetId:     task.TweetId,
				Enable:      task.Enable,
			})
		}
	}
	return
}
