package dao

import (
	"sdk_api/constant"
	dbModel "sdk_api/model/db"
	"sdk_api/util"
	"time"

	"gorm.io/gorm"
	"gorm.io/gorm/clause"
)

func (d *Dao) InitTaskAction() (err error) {
	for platform, actions := range constant.TaskAction {
		for _, action := range actions {
			t := &dbModel.TaskAction{
				Id:       util.GenFlakeID(),
				Platform: platform,
				Action:   action,
			}
			err = d.db.Clauses(clause.OnConflict{DoNothing: true}).Create(t).Error
			if err != nil {
				return err
			}
		}
	}
	return err
}

func (d *Dao) CreateProject(p *dbModel.Project) (err error) {
	return d.db.Create(p).Error
}

func (d *Dao) GetProjectList(page, pageSize int) (list []*dbModel.Project, err error) {
	return list, d.db.Limit(pageSize).Offset((page - 1) * pageSize).Find(&list).Error
}

func (d *Dao) GetProject(id int) (p *dbModel.Project, err error) {
	p = &dbModel.Project{}
	err = d.db.Where("id = ?", id).First(&p).Error
	if err == gorm.ErrRecordNotFound {
		return nil, nil
	}
	return p, err
}

func (d *Dao) CreateGroup(g *dbModel.TaskGroup) (err error) {
	return d.db.Create(g).Error
}

func (d *Dao) GetGroupList(page, pageSize int) (list []*dbModel.TaskGroup, err error) {
	return list, d.db.Limit(pageSize).Offset((page - 1) * pageSize).Find(&list).Error
}

func (d *Dao) GetGroup(id int) (g *dbModel.TaskGroup, err error) {
	g = &dbModel.TaskGroup{}
	err = d.db.Where("id = ?", id).First(&g).Error
	if err == gorm.ErrRecordNotFound {
		return nil, nil
	}
	return g, err
}

func (d *Dao) GetGroupTasks(gid int) (list []*dbModel.Task, err error) {
	return list, d.db.Where("group_id = ?", gid).Find(&list).Error
}

func (d *Dao) CreateGroupTask(gt *dbModel.Task) (err error) {
	return d.db.Create(gt).Error
}

func (d *Dao) GetTaskDetail(tid int) (t *dbModel.Task, err error) {
	t = &dbModel.Task{}
	err = d.db.Where("id = ?", tid).First(&t).Error
	if err == gorm.ErrRecordNotFound {
		return nil, nil
	}
	return t, err

}
func (d *Dao) IsDailyTask(tid int) (ok bool, err error) {
	var count int64
	err = d.db.Model(&dbModel.Task{}).Where("id = ? and daily = ?", tid, 1).Count(&count).Error
	if err != nil {
		return false, err
	}
	return count > 0, nil
}

// IsTaskDone 是否任务已完成
func (d *Dao) IsTaskDone(tid int, userId string) (ok bool, err error) {
	isDailyTask, err := d.IsDailyTask(tid)
	if err != nil {
		return false, err
	}

	var count int64
	if !isDailyTask {
		err = d.db.Model(&dbModel.TaskHistory{}).Where("task_id = ? and user_id = ?", tid, userId).Count(&count).Error
		return count > 0, err
	}

	err = d.db.Model(&dbModel.TaskHistory{}).
		Where("task_id = ? and user_id = ? and created_at >= ?", tid, userId, time.Now().UTC().Truncate(24*time.Hour)).
		Count(&count).Error
	return count > 0, err
}

func (d *Dao) CreateTaskHistory(taskId int, userId string, isDailyTask bool) (err error) {
	// 使用行锁，防止并发
	tx := d.db.Begin()
	defer func() {
		if err != nil {
			tx.Rollback()
		} else {
			tx.Commit()
		}
	}()
	temp := &dbModel.TaskHistory{}
	if !isDailyTask {
		err = tx.Set("gorm:query_option", "FOR UPDATE NOWAIT").Where("task_id = ? and user_id = ?", taskId, userId).First(temp).Error
		if err == gorm.ErrRecordNotFound {
			err = tx.Create(&dbModel.TaskHistory{
				Id:     util.GenFlakeID(),
				TaskId: taskId,
				UserId: userId,
			}).Error
			if err != nil {
				return err
			}
		}
		return err
	}

	err = tx.Set("gorm:query_option", "FOR UPDATE NOWAIT").Where("task_id = ? and user_id = ? and created_at >= ?", taskId, userId, time.Now().UTC().Truncate(24*time.Hour)).First(temp).Error
	if err == gorm.ErrRecordNotFound {
		err = tx.Create(&dbModel.TaskHistory{
			Id:     util.GenFlakeID(),
			TaskId: taskId,
			UserId: userId,
		}).Error
		if err != nil {
			return err
		}
	}
	return err
}
