package dao

import (
	"fmt"
	"os"
	"taskcenter/config"
	dbModel "taskcenter/model/db"
	"time"

	"github.com/supabase-community/supabase-go"
	"gorm.io/driver/postgres"
	"gorm.io/gorm"
	"gorm.io/gorm/logger"
	"gorm.io/gorm/schema"
)

type Dao struct {
	c    *config.Config
	db   *gorm.DB
	supa *supabase.Client
}

func New(_c *config.Config) (dao *Dao, err error) {
	dao = &Dao{
		c: _c,
	}

	// dsn := fmt.Sprintf("host=%s user=%s password=%s dbname=%s port=%d sslmode=verify-ca sslrootcert=%s",
	dsn := fmt.Sprintf("host=%s user=%s password=%s dbname=%s port=%d",
		_c.PGSQL.Host, _c.PGSQL.User, _c.PGSQL.Password, _c.PGSQL.Database, _c.PGSQL.Port,
	)

	if _c.PGSQL.CertFile != "" {
		dsn = fmt.Sprintf("%s sslmode=require sslrootcert=%s", dsn, _c.PGSQL.CertFile)
	}

	lgr := logger.Default
	if _c.PGSQL.EnableLog {
		lgr = logger.Default.LogMode(logger.Info)
	}

	dao.db, err = gorm.Open(postgres.Open(dsn), &gorm.Config{
		NamingStrategy: schema.NamingStrategy{
			SingularTable: true,
		},
		DisableForeignKeyConstraintWhenMigrating: true, // 停用外键约束
		Logger:                                   lgr,
	})
	if err != nil {
		return
	}
	sqlDB, err := dao.db.DB()
	if err != nil {
		return
	}
	sqlDB.SetMaxOpenConns(_c.PGSQL.MaxConn)
	sqlDB.SetMaxIdleConns(_c.PGSQL.MaxIdleConn)
	sqlDB.SetConnMaxIdleTime(time.Hour)

	if os.Getenv("MIGRATE") == "true" {
		err = dao.db.AutoMigrate(&dbModel.TaskAction{}, &dbModel.TaskGroup{}, &dbModel.Task{}, &dbModel.TaskHistory{}, &dbModel.AdminUser{})
		if err != nil {
			return
		}
		if err = dao.InitTaskAction(); err != nil {
			panic(err)
		}
	}

	return dao, nil
}
