package dao

import (
	"bytes"
	"encoding/json"
	"errors"
	"fmt"
	"strconv"
	"strings"
	"taskcenter/constant"
	"time"

	log "github.com/sirupsen/logrus"
	"github.com/tidwall/gjson"
)

type TwitterConfig struct {
	APIKey            string
	APISecret         string
	AccessToken       string
	AccessTokenSecret string
	TwitterToken      string
}

func (d *Dao) GetTwitterInfo(config TwitterConfig) (twitterUserId int, twitterHandle, twitterName string, err error) {
	body := map[string]string{
		"api_key":             config.APIKey,
		"api_key_secret":      config.APISecret,
		"access_token":        config.AccessToken,
		"access_token_secret": config.AccessTokenSecret,
		"token":               config.TwitterToken,
	}
	buf := new(bytes.Buffer)
	err = json.NewEncoder(buf).Encode(body)
	if err != nil {
		return
	}
	resp, err := httpPost(fmt.Sprintf("%s/project", strings.TrimRight(d.c.TwitterTask.URL, "/")), buf)
	if err != nil {
		return
	}

	respTemp := struct {
		Code int    `json:"code"`
		Msg  string `json:"msg"`
		Data struct {
			UserID   string `json:"user_id"`
			Username string `json:"username"`
			Name     string `json:"name"`
		} `json:"data"`
	}{}

	err = json.Unmarshal(resp, &respTemp)
	if err != nil {
		return
	}

	// if respTemp.Code != 200 && respTemp.Msg != "task already existed" {
	if respTemp.Code != 200 && respTemp.Msg != "already existed" {
		err = errors.New(string(resp))
		return
	}

	userId, _ := strconv.Atoi(respTemp.Data.UserID)

	return userId, respTemp.Data.Username, respTemp.Data.Name, nil
}

func (d *Dao) DoTweetTask(twitterUserId int, tweetId int, twitterHandle, action string, start bool, config TwitterConfig) (err error) {
	body := map[string]interface{}{
		"user_id":   fmt.Sprintf("%d", twitterUserId),
		"task_id":   fmt.Sprintf("%d", tweetId),
		"task_type": action,
		"config": map[string]string{
			"api_key":             config.APIKey,
			"api_key_secret":      config.APISecret,
			"access_token":        config.AccessToken,
			"access_token_secret": config.AccessTokenSecret,
			"token":               config.TwitterToken,
		},
	}

	if twitterHandle != "" && action == constant.TwitterAPIActionFollow {
		body["task_id"] = twitterHandle
	}

	buf := new(bytes.Buffer)
	err = json.NewEncoder(buf).Encode(body)
	if err != nil {
		return
	}

	log.Info(buf.String())

	url := fmt.Sprintf("%s/task/add", strings.TrimRight(d.c.TwitterTask.URL, "/"))
	if !start {
		url = fmt.Sprintf("%s/task/stop", strings.TrimRight(d.c.TwitterTask.URL, "/"))
	}

	resp, err := httpPost(url, buf)
	if err != nil {
		return
	}

	if gjson.Get(string(resp), "code").Int() != 200 && gjson.Get(string(resp), "msg").String() != "task already existed" {
		return errors.New(string(resp))
	}

	return
}

func (d *Dao) CheckTwitterFollow(twitterUserHandle string, followerId string) (ok bool, err error) {
	url := fmt.Sprintf("%s/verify/follower?task_id=%s&follower_id=%s", strings.TrimSuffix(d.c.TwitterTask.URL, "/"), twitterUserHandle, followerId)
	log.WithField("url", url).Debug("check tweet follow")
	data, err := httpGet(url)
	if err != nil {
		return false, err
	}

	if gjson.Get(string(data), "code").Int() != 200 {
		return false, errors.New(string(data))
	}
	return gjson.Get(string(data), "data.ok").Bool(), nil
}

func (d *Dao) CheckTwitterLike(tweetId int, userId string, begin, end int) (ok bool, err error) {
	url := fmt.Sprintf("%s/verify/like?tweet_id=%d&user_id=%s", strings.TrimSuffix(d.c.TwitterTask.URL, "/"), tweetId, userId)
	if begin != 0 && end != 0 {
		url += fmt.Sprintf("&begin=%s&end=%s", time.Unix(int64(begin), 0).Format(time.RFC3339), time.Unix(int64(end), 0).Format(time.RFC3339))
	}

	log.WithField("url", url).Debug("check tweet like")
	data, err := httpGet(url)
	if err != nil {
		return false, err
	}

	if gjson.Get(string(data), "code").Int() != 200 {
		return false, errors.New(string(data))
	}
	return gjson.Get(string(data), "data.ok").Bool(), nil
}

func (d *Dao) CheckTwitterRetweet(tweetId int, userId string, begin, end int) (ok bool, err error) {
	url := fmt.Sprintf("%s/verify/retweeter?tweet_id=%d&retweeter_id=%s", strings.TrimSuffix(d.c.TwitterTask.URL, "/"), tweetId, userId)
	if begin != 0 && end != 0 {
		url += fmt.Sprintf("&begin=%s&end=%s", time.Unix(int64(begin), 0).Format(time.RFC3339), time.Unix(int64(end), 0).Format(time.RFC3339))
	}

	log.WithField("url", url).Debug("check tweet retweet")
	data, err := httpGet(url)
	if err != nil {
		return false, err
	}

	if gjson.Get(string(data), "code").Int() != 200 {
		return false, errors.New(string(data))
	}
	return gjson.Get(string(data), "data.ok").Bool(), nil
}
