package middleware

import (
	"bytes"
	"context"
	"fmt"
	"io"
	"net/http"
	"strings"
	"taskcenter/config"
	"taskcenter/constant"
	"taskcenter/dao"
	"time"

	"github.com/gin-gonic/gin"
	log "github.com/sirupsen/logrus"
	"github.com/tidwall/gjson"
)

func JWTAuthMiddleware(d *dao.Dao, cfg config.SupabaseConfig, needAdmin bool) gin.HandlerFunc {
	return func(c *gin.Context) {
		tokenString := c.GetHeader("Authorization")
		log.Debugln(tokenString)
		if tokenString == "" || len(tokenString) < 7 {
			c.JSON(200, gin.H{
				"code": 1,
				"msg":  "invalid token",
				"data": "",
			})
			c.Abort()
			return
		}
		timeoutCtx, cancel := context.WithTimeout(context.Background(), time.Second*10)
		defer cancel()
		req, err := http.NewRequestWithContext(timeoutCtx, http.MethodGet, fmt.Sprintf("%s/auth/v1/user", strings.TrimRight(cfg.AuthURL, "/")), bytes.NewBuffer(nil))
		if err != nil {
			log.WithError(err).Error("auth middleware new request")
			c.JSON(200, gin.H{
				"code": 1,
				"msg":  constant.InternalError,
				"data": "",
			})
			c.Abort()
			return
		}

		req.Header.Set("apiKey", cfg.APIKey)
		req.Header.Set("Authorization", tokenString)
		resp, err := http.DefaultClient.Do(req)
		if err != nil {
			log.WithError(err).Error("auth middleware do request")
			c.JSON(200, gin.H{
				"code": 1,
				"msg":  constant.InternalError,
				"data": "",
			})
			c.Abort()
			return
		}
		defer resp.Body.Close()
		data, err := io.ReadAll(resp.Body)
		if err != nil {
			log.WithError(err).Error("auth middleware read all")
			c.JSON(200, gin.H{
				"code": 1,
				"msg":  constant.InternalError,
				"data": "",
			})
			c.Abort()
			return
		}

		userId := gjson.Get(string(data), "data.user.id").String()
		isAnonymous := gjson.Get(string(data), "data.user.is_anonymous").Bool()
		if userId == "" || isAnonymous {
			log.WithField("resp", string(data)).Warn("parse data from bearer token")
			c.JSON(200, gin.H{
				"code": 1,
				"msg":  "invalid token",
				"data": "",
			})
			c.Abort()
			return
		}

		c.Set("userId", userId)

		if needAdmin {
			ok, err := d.IsAdminUser(userId)
			if err != nil {
				log.WithError(err).Error("auth is admin")
				c.JSON(200, gin.H{
					"code": 1,
					"msg":  constant.InternalError,
					"data": "",
				})
				c.Abort()
				return
			}
			if !ok {
				c.JSON(200, gin.H{
					"code": 1,
					"msg":  "no access permissions",
					"data": "",
				})
				c.Abort()
				return
			}
		}

		c.Next()
	}
}
