package server

import (
	"strconv"
	"taskcenter/constant"
	apiModel "taskcenter/model/api"

	"github.com/gin-gonic/gin"
)

func createTask(c *gin.Context) {
	req := &apiModel.CreateTaskRequest{}
	if err := c.ShouldBindJSON(req); err != nil {
		c.JSON(200, withError(constant.InvalidParam))
		return
	}
	req.UserId = c.GetString("userId")

	ok, err := srv.CheckGroup(req.GroupId, req.UserId)
	if err != nil {
		c.JSON(200, withError(constant.InternalError))
		return
	}
	if !ok {
		c.JSON(200, withError(constant.InvalidParam))
		return
	}

	taskId, err := srv.CreateTask(req)
	if err != nil {
		c.JSON(200, withError(constant.InternalError))
		return
	}
	c.JSON(200, withSuccess(gin.H{"taskId": taskId}))
}

func editTask(c *gin.Context) {
	req := &apiModel.EditTaskRequest{}
	if err := c.ShouldBindJSON(req); err != nil {
		c.JSON(200, withError(constant.InvalidParam))
		return
	}
	req.UserId = c.GetString("userId")

	userId, err := srv.GetTaskOwner(req.TaskId, req.GroupId)
	if err != nil {
		c.JSON(200, withError(constant.InternalError))
		return
	}

	if userId != req.UserId {
		c.JSON(200, withError("permission denied"))
		return
	}

	if req.Delete {
		req.Enable = false
	}

	err = srv.EditTask(req)
	if err != nil {
		c.JSON(200, withError(constant.InternalError))
		return
	}
	c.JSON(200, withSuccess(""))
}

func checkTask(c *gin.Context) {
	_taskId := c.Param("tid")
	taskId, _ := strconv.Atoi(_taskId)
	// userId := c.Query("userId")
	userId := c.GetString("userId")

	if taskId == 0 || userId == "" {
		c.JSON(200, withError(constant.InvalidParam))
		return
	}

	status, err := srv.GetTaskResult(taskId, userId)
	if err != nil {
		c.JSON(200, withError(constant.InternalError))
		return
	}

	c.JSON(200, withSuccess(gin.H{"status": status}))
}

// submitTask 提交任务，等待后台检查
func submitTask(c *gin.Context) {
	_taskId := c.Param("tid")
	taskId, _ := strconv.Atoi(_taskId)
	// userId := c.Query("userId")
	userId := c.GetString("userId")

	if taskId == 0 || userId == "" {
		c.JSON(200, withError(constant.InvalidParam))
		return
	}

	task, err := srv.GetTaskDetail(taskId)
	if err != nil {
		c.JSON(200, withError(constant.InternalError))
		return
	}

	if task == nil {
		c.JSON(200, withError("task not found"))
		return
	}

	status, err := srv.GetTaskResult(taskId, userId)
	if err != nil {
		c.JSON(200, withError(constant.InternalError))
		return
	}

	if status == constant.TaskHistoryStatusPending {
		c.JSON(200, withError("task already submitted"))
		return
	}

	if status == constant.TaskHistoryStatusSuccess {
		c.JSON(200, withError("task already done"))
		return
	}

	_, err = srv.SubmitTask(taskId, userId, task.Daily)
	if err != nil {
		c.JSON(200, withError(constant.InternalError))
		return
	}

	c.JSON(200, withSuccess(gin.H{}))
}
