package main

import (
	"encoding/json"
	"fmt"
	"log/slog"

	// "github.com/gofiber/fiber/v2"
	// "github.com/gofiber/fiber/v2/middleware/cors"
	// "github.com/gofiber/swagger"

	"github.com/gofiber/fiber/v2"

	//"github.com/gofiber/swagger"

	_ "code.wuban.net.cn/odysseus/twitter_syncer/docs"
	// docs are generated by Swag CLI, you have to import them.
	// replace with your own docs folder, usually "github.com/username/reponame/docs"
	//_ "github.com/gofiber/swagger/example/docs"
)

type Res struct {
	Code int64  `json:"code"`
	Msg  string `json:"msg"`
}

type VerifyRes struct {
	Code int64  `json:"code"`
	Msg  string `json:"msg"`
	Data struct {
		Ok bool `json:"ok"`
	} `json:"data"`
}

type ProjectRes struct {
	Code int64    `json:"code"`
	Msg  string   `json:"msg"`
	Data UserInfo `json:"data"`
}

/*

{
  "user_id": "1570057485914087429",
  "project": "OnlyDD_D",
  "task_type": "follow",
  "start": true
}

*/

type ProjectReq struct {
	Config
	Project string `json:"project"`
}

// GetConfigOwner godoc
// @Summary GetConfigOwner
// @Description get config owner info.
// @Tags config
// @Accept  json
// @Produce  json
// @Param config body Config true "config"
// @Success 200 {object} ProjectRes
// @Failure 400 {object} Res
// @Failure 500 {object} Res
// @Router /apikey/owner [get]
func GetConfigOwner(c *fiber.Ctx) error {

	slog.Info(c.Route().Path, "body", string(c.Request().Body()))

	reqCfg := Config{}

	if err := json.Unmarshal(c.Request().Body(), &reqCfg); err != nil {
		slog.Error("json.Unmarshal(c.Request().Body(), &req)", "err", err.Error())
		return c.JSON(Res{
			Code: 500,
			Msg:  err.Error(),
		})
	}

	slog.Info("get config owner info", "config", reqCfg)
	user, err := NewClient(reqCfg, nil).MyInfo()
	if err != nil {
		slog.Error("MyInfo", "err", err.Error())
		return c.JSON(Res{
			Code: 500,
			Msg:  err.Error(),
		})
	}
	return c.JSON(ProjectRes{
		Code: 200,
		Data: user,
	})

}

// add comment to generate swagger docs, example as below.
// GetUser godoc
// @Summary Get a user
// @Description get user by ID
// @Tags users
// @Accept  json
// @Produce  json
// @Param id path int true "User ID"
// @Success 200 {object} models.User
// @Failure 400 {object} models.HTTPError
// @Failure 404 {object} models.HTTPError
// @Failure 500 {object} models.HTTPError
// @Router /users/{id} [get]

// Project godoc
// @Summary Project
// @Description add project info.
// @Tags project
// @Accept  json
// @Produce  json
// @Param project body ProjectReq true "project"
// @Success 200 {object} ProjectRes
// @Failure 400 {object} Res
// @Failure 500 {object} Res
// @Router /project [post]
func Project(c *fiber.Ctx) error {

	slog.Info(c.Route().Path, "body", string(c.Request().Body()))

	req := ProjectReq{}

	if err := json.Unmarshal(c.Request().Body(), &req); err != nil {
		slog.Error("json.Unmarshal(c.Request().Body(), &req)", "err", err.Error())
		return c.JSON(ProjectRes{
			Code: 500,
			Msg:  err.Error(),
		})
	}

	slog.Info("cfg", "project", req.Project, "ApiKey", req.ApiKey, "req.ApiKeySecrect", req.ApiKeySecrect, "req.AccessToken", req.AccessToken, "req.AccessTokenSecret", req.AccessTokenSecret, "req.Token", req.Token)

	data, ok, err := QueryProjectByKeysAndToken(req)

	if err != nil {
		slog.Error("QueryProject", "err", err.Error())
		return c.JSON(ProjectRes{
			Code: 500,
			Msg:  err.Error(),
		})
	}

	if ok {
		return c.JSON(ProjectRes{
			Code: 500,
			Msg:  "already existed",
			Data: data,
		})

	}

	user, err := NewClient(req.Config, nil).MyInfo()
	if err != nil {
		slog.Error("MyInfo", "err", err.Error())
		return c.JSON(ProjectRes{
			Code: 500,
			Msg:  err.Error(),
		})
	}
	if err := AddOrUpdateProject(req, user); err != nil {
		slog.Error("insert db ", "err", err.Error())
		return c.JSON(ProjectRes{
			Code: 500,
			Msg:  err.Error(),
		})
	}

	return c.JSON(ProjectRes{
		Code: 200,
		Data: user,
	})
}

type AddTaskReq struct {
	User      string `json:"user_id"`
	TaskType  string `json:"task_type"`
	TaskId    string `json:"task_id"`
	ApiConfig Config `json:"config"`
}

// TaskAdd godoc
// @Summary TaskAdd
// @Description add task info.
// @Tags task
// @Accept  json
// @Produce  json
// @Param task body AddTaskReq true "task"
// @Success 200 {object} Res
// @Failure 400 {object} Res
// @Failure 500 {object} Res
// @Router /task/add [post]
func TaskAdd(c *fiber.Ctx) error {
	slog.Info(c.Route().Path, "body", string(c.Request().Body()))

	req := AddTaskReq{}

	if err := json.Unmarshal(c.Request().Body(), &req); err != nil {
		slog.Error("json.Unmarshal(c.Request().Body(), &req)", "err", err.Error())
		return c.JSON(Res{
			Code: 500,
			Msg:  err.Error(),
		})
	}

	slog.Info(c.Route().Path, "user", req.User, "TaskType", req.TaskType, "TaskId", req.TaskId, "config", req.ApiConfig)

	if req.TaskType == "" || req.TaskId == "" {
		return c.JSON(Res{
			Code: 500,
			Msg:  "must provide TaskId and TaskType",
		})
	}
	// check the config is valid or not.
	user, err := NewClient(req.ApiConfig, nil).MyInfo()
	if err != nil {
		return c.JSON(Res{
			Code: 500,
			Msg:  "get api owner info failed " + err.Error(),
		})
	}
	if user.UserId != req.User {
		return c.JSON(Res{
			Code: 500,
			Msg:  "user_id not match with the api owner",
		})
	}

	// 校验任务 条件是否存在；
	if exist, err := CheckTaskExist(req.User, req.TaskId, req.TaskType); err != nil {
		return c.JSON(Res{
			Code: 500,
			Msg:  "QCheckTask " + req.User,
		})
	} else if exist {
		return c.JSON(Res{
			Code: 500,
			Msg:  "task already existed",
		})
	}

	fc := 0
	if req.TaskType == FollowType { // do some check.
		// follower task count < available account count.
		if passed, err := CheckFollowerTaskAndAccountCount(); err != nil {
			return c.JSON(Res{
				Code: 500,
				Msg:  "CheckFollowerTaskAndAccountCount " + req.User,
			})
		} else if !passed {
			return c.JSON(Res{
				Code: 500,
				Msg:  "CheckFollowerTaskAndAccountCount !ok",
			})
		}

		fc, err = NewFollowerOb().TryProfileFollowerCount(req.TaskId)
		if err != nil {
			return c.JSON(Res{
				Code: 500,
				Msg:  "QCheckTask " + req.User,
			})
		}

	}

	// create a taskJob.
	job := TaskJob{
		Config:        req.ApiConfig,
		Idx:           make([]UserTask, 0),
		UserId:        req.User,
		TaskId:        req.TaskId,
		TaskType:      req.TaskType,
		FollowerCount: fc,
	}

	slog.Info("new", "job-", job.String())

	err = Worker.AddJob(job)
	if err != nil {

		slog.Error(" Worker.AddJob", "err", err.Error())
		return c.JSON(Res{
			Code: 500,
			Msg:  err.Error(),
		})
	}

	slog.Info("add task into db", "user", req.User, "TaskType", req.TaskType, "TaskId", req.TaskId)
	if err = AddTaskInsert(req, fc); err != nil {
		slog.Error("twitter_syncer insert", "err", err.Error())
		return c.JSON(Res{
			Code: 500,
			Msg:  err.Error(),
		})
	}

	return c.JSON(Res{
		Code: 200,
	})
}

type StopTaskReq struct {
	User     string `json:"user_id"`
	TaskType string `json:"task_type"`
	TaskId   string `json:"task_id"`
}

// TaskStop godoc
// @Summary TaskStop
// @Description stop task info.
// @Tags task
// @Accept  json
// @Produce  json
// @Param task body StopTaskReq true "task"
// @Success 200 {object} Res
// @Failure 400 {object} Res
// @Failure 500 {object} Res
// @Router /task/stop [post]
func TaskStop(c *fiber.Ctx) error {

	slog.Info(c.Route().Path, "body", string(c.Request().Body()))

	req := StopTaskReq{}

	if err := json.Unmarshal(c.Request().Body(), &req); err != nil {
		slog.Error("json.Unmarshal(c.Request().Body(), &req)", "err", err.Error())
		return c.JSON(Res{
			Code: 500,
			Msg:  err.Error(),
		})
	}

	slog.Info(c.Route().Path, "user", req.User, "TaskType", req.TaskType, "TaskId", req.TaskId)

	if req.TaskType == "" || req.TaskId == "" {
		return c.JSON(Res{
			Code: 500,
			Msg:  "must provide TaskId and TaskType",
		})
	}

	slog.Info("stop job", "user", req.User, "TaskType", req.TaskType, "TaskId", req.TaskId)

	if err := Worker.StopJob(req); err != nil {
		return c.JSON(Res{
			Code: 500,
			Msg:  err.Error(),
		})
	}

	slog.Info("set stop to true in db", "user", req.User, "TaskType", req.TaskType, "TaskId", req.TaskId)

	err := StopTaskUpdate(req)

	if err != nil {

		slog.Error("twitter_syncer stop", "err", err.Error())
		return c.JSON(Res{
			Code: 500,
			Msg:  err.Error(),
		})
	}

	return c.JSON(Res{
		Code: 200,
	})

}

// VerifyRetweeter godoc
// @Summary VerifyRetweeter
// @Description verify retweeter info.
// @Tags verify
// @Accept  json
// @Produce  json
// @Param tweet_id query string true "tweet_id"
// @Param retweeter_id query string true "retweeter_id"
// @Success 200 {object} VerifyRes
// @Failure 400 {object} Res
// @Failure 500 {object} Res
// @Router /verify/retweeter [get]
func VerifyRetweeter(c *fiber.Ctx) error {

	tweetId := c.Query("tweet_id")
	retweeterId := c.Query("retweeter_id")

	// beginTime := c.Query("begin_time")
	// endTime := c.Query("end_time")

	if len(tweetId) == 0 || len(retweeterId) == 0 {
		slog.Error("VerifyFollower", "tweetId", tweetId, "retweeterId", retweeterId)
		return c.JSON(Res{
			Code: 500,
			Msg:  fmt.Sprintf("must provide tweetId [%v] and retweeterId [%v]", tweetId, retweeterId),
		})
	}

	slog.Info(c.Route().Path, "tweetId", tweetId, "retweeterId", retweeterId)

	ok, err := VerifyRetweeterInDb(tweetId, retweeterId)

	if err != nil {
		slog.Error("VerifyRetweeter", "tweetId", tweetId, "retweeterId", retweeterId, "err", err.Error())
		return c.JSON(Res{
			Code: 500,
			//Msg:  fmt.Sprint("VerifyFollowerInDb", "tweetId", tweetId, "retweeterId", retweeterId, "err", err.Error()),
			Msg: fmt.Sprintf("VerifyRetweeter  tweetId   %v retweeterId %v  err %v", tweetId, retweeterId, err.Error()),
		})
	}

	return c.JSON(VerifyRes{
		Code: 200,
		Data: struct {
			Ok bool "json:\"ok\""
		}{
			Ok: ok,
		},
	})

}

// VerifyFollower godoc
// @Summary VerifyFollower
// @Description verify follower info.
// @Tags verify
// @Accept  json
// @Produce  json
// @Param task_id query string true "task_id"
// @Param follower_id query string true "follower_id"
// @Param follower_username query string true "follower_username"
// @Success 200 {object} VerifyRes
// @Failure 400 {object} Res
// @Failure 500 {object} Res
// @Router /verify/follower [get]
func VerifyFollower(c *fiber.Ctx) error {

	userId := c.Query("task_id")
	followerId := c.Query("follower_id")

	if len(userId) == 0 || len(followerId) == 0 {
		slog.Error("VerifyFollower", "userId", userId, "followerId", followerId)
		return c.JSON(Res{
			Code: 500,
			Msg:  fmt.Sprintf("must provide userId [%v] and followerId [%v]", userId, followerId),
		})
	}

	followerUserName := c.Query("follower_username")

	slog.Info(c.Route().Path, "user_id", userId, "followerId", followerId, "followerUserName", followerUserName)

	ok, err := VerifyFollowerInDb(userId, followerId)

	if err != nil {
		slog.Error("VerifyFollowerInDb", "userId", userId, "followerId", followerId, "err", err.Error())
		return c.JSON(Res{
			Code: 500,
			Msg:  fmt.Sprintf("VerifyFollowerInDb  userId   %v followerId %v  err %v", userId, followerId, err.Error()),
		})
	}

	return c.JSON(VerifyRes{
		Code: 200,
		Data: struct {
			Ok bool "json:\"ok\""
		}{
			Ok: ok,
		},
	})

}

// VerifyLike godoc
// @Summary VerifyLike
// @Description verify like info.
// @Tags verify
// @Accept  json
// @Produce  json
// @Param tweet_id query string true "tweet_id"
// @Param user_id query string true "user_id"
// @Success 200 {object} VerifyRes
// @Failure 400 {object} Res
// @Failure 500 {object} Res
// @Router /verify/like [get]
func VerifyLike(c *fiber.Ctx) error {

	tweetId := c.Query("tweet_id")
	retweeterId := c.Query("user_id")

	if len(tweetId) == 0 || len(retweeterId) == 0 {
		slog.Error("VerifyFollower", "tweetId", tweetId, "user_id", retweeterId)
		return c.JSON(Res{
			Code: 500,
			Msg:  fmt.Sprintf("must provide tweetId [%v] and user_id [%v]", tweetId, retweeterId),
		})
	}

	slog.Info(c.Route().Path, "tweetId", tweetId, "user_id", retweeterId)

	ok, err := VerifyLikeInDb(tweetId, retweeterId)

	if err != nil {
		slog.Error("VerifyRetweeter", "tweetId", tweetId, "user_id", retweeterId, "err", err.Error())
		return c.JSON(Res{
			Code: 500,
			//Msg:  fmt.Sprint("VerifyFollowerInDb", "tweetId", tweetId, "retweeterId", retweeterId, "err", err.Error()),
			Msg: fmt.Sprintf("VerifyRetweeter  tweetId   %v user_id %v  err %v", tweetId, retweeterId, err.Error()),
		})
	}

	return c.JSON(VerifyRes{
		Code: 200,
		Data: struct {
			Ok bool "json:\"ok\""
		}{
			Ok: ok,
		},
	})

}
