package swarm

import (
	"fmt"
	"github.com/g8rswimmer/go-twitter/v2"
	"github.com/xueqianLu/twitter-bee/client"
	"sync"
)

type ClientWithRateLimiter struct {
	*client.BeeClient
}

type Swarm struct {
	clients    map[string]*ClientWithRateLimiter
	mu         sync.Mutex
	profilemu  sync.Mutex
	followermu sync.Mutex
}

var (
	gSwarm *Swarm
)

func GetSwarm() *Swarm {
	if gSwarm == nil {
		gSwarm, _ = InitSwarm(nil)
	}
	return gSwarm
}

func InitSwarm(initialBees []string) (*Swarm, error) {
	s := &Swarm{
		clients: make(map[string]*ClientWithRateLimiter),
	}
	for _, bee := range initialBees {
		s.AddClient(bee)
	}
	gSwarm = s
	return s, nil
}

func (s *Swarm) copyedClients() map[string]*ClientWithRateLimiter {
	s.mu.Lock()
	defer s.mu.Unlock()
	clients := make(map[string]*ClientWithRateLimiter)
	for k, v := range s.clients {
		clients[k] = v
	}
	return clients
}

func (s *Swarm) AddClient(url string) {
	s.mu.Lock()
	defer s.mu.Unlock()
	cli := new(ClientWithRateLimiter)
	cli.BeeClient = client.NewBeeClient(url)

	s.clients[url] = cli
}

func (s *Swarm) RemoveClient(url string) {
	s.mu.Lock()
	defer s.mu.Unlock()
	delete(s.clients, url)
}

func (s *Swarm) GetUserProfile(name string) (string, int, error) {
	clients := s.copyedClients()
	s.profilemu.Lock()
	defer s.profilemu.Unlock()

	for _, cli := range clients {
		res, err := cli.GetUserProfile(name)
		if err == nil {
			fmt.Println("get user profile", res)
			return res.Id, res.Follower, nil
		} else {
			fmt.Println("get user profile failed with err", err.Error())
		}
	}
	return "", 0, fmt.Errorf("can not get the %v profile", name)
}

func (s *Swarm) GetFollowerList(user string, id string, cursor string) ([]*twitter.UserObj, string, *twitter.RateLimit, error) {
	clients := s.copyedClients()

	s.followermu.Lock()
	defer s.followermu.Unlock()

	for _, cli := range clients {
		res, err := cli.GetFollowerList(user, id, cursor)
		if err == nil {
			list := make([]*twitter.UserObj, 0, len(res.List))
			for _, u := range res.List {
				list = append(list, &twitter.UserObj{
					ID:       u.ID,
					Name:     u.Name,
					UserName: u.UserName,
				})
			}
			return list, res.Next, nil, nil
		} else {
			fmt.Println("get follower list failed with err", err.Error())
		}
	}
	return nil, "", nil, fmt.Errorf("can not get the %v follower list", user)
}
