package acmanager

import (
	"encoding/json"
	"github.com/gofiber/fiber/v2/log"
	"github.com/supabase-community/supabase-go"
	"net/http"
	"time"
)

type Account struct {
	Username string `json:"username"`
	Password string `json:"password"`
	Email    string `json:"email"`
	// Cookies   []byte `json:"cookies"`
	F2A string `json:"two_fa_pk"`

	Cookies   []*http.Cookie `json:"cookies"`
	Available bool           `json:"available"`
}

type Manager struct {
	quit              chan struct{}
	client            *supabase.Client
	availableUser     map[string]Account
	userBlockedRecord map[string]time.Time
	warningHistory    []WarningRecord
}

func NewManager(client *supabase.Client) *Manager {
	m := &Manager{
		quit:           make(chan struct{}),
		availableUser:  make(map[string]Account),
		warningHistory: make([]WarningRecord, 0),
		client:         client,
	}
	m.warningHistory = append(m.warningHistory, WarningRecord{
		IsWarning: false,
		Time:      time.Now(),
	})

	return m
}

func (m *Manager) report(current int) {
	if current < warningCount {
		// check need to report.
		last := m.warningHistory[len(m.warningHistory)-1]
		// if last is not warning or last warning passed 5 minutes, send warning email.
		if !last.IsWarning || (last.IsWarning && time.Now().After(last.Time.Add(time.Minute*5))) {
			// send warning email.
			if err := Warning(current); err != nil {
				// log error.
				log.Errorf("send warning email error: %s", err.Error())
			} else {
				m.warningHistory = append(m.warningHistory, WarningRecord{
					IsWarning: true,
					Time:      time.Now(),
				})
			}
		}
	} else {
		last := m.warningHistory[len(m.warningHistory)-1]
		if last.IsWarning {
			// send success email.
			if err := Recovered(current); err != nil {
				// log error.
				log.Errorf("send success email error: %s", err.Error())
			} else {
				m.warningHistory = append(m.warningHistory, WarningRecord{
					IsWarning: false,
					Time:      time.Now(),
				})
			}
		}
	}
	if len(m.warningHistory) > 10 {
		old := m.warningHistory
		m.warningHistory = make([]WarningRecord, 0)
		m.warningHistory = append(m.warningHistory, old[len(old)-1])
	}
}

func (m *Manager) loop() {
	// loop check available account count.
	tc := time.NewTicker(time.Minute * 1)
	defer tc.Stop()
	for {
		select {
		case <-tc.C:
			all, err := GetAllAccounts(m.client)
			if err == nil {
				available := make([]Account, 0)
				unavailable := make([]Account, 0)
				for _, v := range all {
					if v.Available {
						available = append(available, v)
					} else {
						unavailable = append(unavailable, v)
					}
				}
				m.report(len(available))
			}
			tc.Reset(time.Minute * 3)
		case <-m.quit:
			return
		}
	}
}

func (m *Manager) Start() {
	go m.loop()
}

func (m *Manager) Stop() {
	close(m.quit)
}

func GetAllAccounts(client *supabase.Client) ([]Account, error) {
	data, count, err := client.From("accounts").Select("*", "exact", false).Execute()
	if err != nil {
		return nil, err
	}

	_ = count

	res := make([]Account, 0, count)
	if err := json.Unmarshal(data, &res); err != nil {
		return nil, err
	}

	return res, nil

}
