package acmanager

import (
	"fmt"
	"gopkg.in/gomail.v2"
	"os"
	"strconv"
	"strings"
	"time"
)

const (
	warningEmailTemplate = `
Hi there,
    Current available account is %d, it is less than %d, please check it.
`
	recoversEmailTemplate = `
Hi there,
	Current available account is %d, warning cancelled.
`
	warningCount = 5

	ENV_ALERT_RECEIVERS = "TW_ALERT_RECEIVERS"
	ENV_ALERT_EMAIL     = "TW_ALERT_EMAIL"
	ENV_EMAIL_SERVER    = "TW_EMAIL_SERVER"
	ENV_EMAIL_PORT      = "TW_EMAIL_PORT"
	ENV_EMAIL_USER      = "TW_EMAIL_USER"
	ENV_EMAIL_PASSWD    = "TW_EMAIL_PASSWD"
)

type WarningRecord struct {
	IsWarning bool
	Time      time.Time
}

type Environment struct {
	AlertReceivers []string
	AlertEmail     string
	EmailServer    string
	EmailPort      int
	EmailUser      string
	EmailPasswd    string
}

func (e Environment) Available() bool {
	if len(e.AlertReceivers) == 0 || e.AlertEmail == "" || e.EmailServer == "" || e.EmailPort == 0 || e.EmailUser == "" || e.EmailPasswd == "" {
		return false
	}
	return true
}

func getEnv() Environment {
	// get env from os.
	env := Environment{}
	if receiver, ok := os.LookupEnv(ENV_ALERT_RECEIVERS); ok {
		rs := strings.Split(receiver, ",")
		env.AlertReceivers = rs
	}
	if email, ok := os.LookupEnv(ENV_ALERT_EMAIL); ok {
		env.AlertEmail = email
	}
	if server, ok := os.LookupEnv(ENV_EMAIL_SERVER); ok {
		env.EmailServer = server
	}
	if port, ok := os.LookupEnv(ENV_EMAIL_PORT); ok {
		env.EmailPort, _ = strconv.Atoi(port)
	}
	if user, ok := os.LookupEnv(ENV_EMAIL_USER); ok {
		env.EmailUser = user
	}
	if passwd, ok := os.LookupEnv(ENV_EMAIL_PASSWD); ok {
		env.EmailPasswd = passwd
	}
	return env
}

func Warning(count int) error {
	// send email with warning template.
	env := getEnv()
	if env.Available() {
		// send warning email.
		msg := MsgInfo{
			From:    env.AlertEmail,
			To:      env.AlertReceivers,
			Title:   "New Warning",
			Content: fmt.Sprintf(warningEmailTemplate, count, warningCount),
		}
		server := ServerInfo{
			Server:   env.EmailServer,
			Port:     env.EmailPort,
			Username: env.EmailUser,
			Passwd:   env.EmailPasswd,
		}
		return sendMail(msg, server)
	} else {
		return fmt.Errorf("email env is not available")
	}
}

func Recovered(count int) error {
	// send email with warning template.
	env := getEnv()
	if env.Available() {
		// send warning email.
		msg := MsgInfo{
			From:    env.AlertEmail,
			To:      env.AlertReceivers,
			Title:   "Warning Cancelled",
			Content: fmt.Sprintf(recoversEmailTemplate, count),
		}
		server := ServerInfo{
			Server:   env.EmailServer,
			Port:     env.EmailPort,
			Username: env.EmailUser,
			Passwd:   env.EmailPasswd,
		}
		return sendMail(msg, server)
	} else {
		return fmt.Errorf("email env is not available")
	}
}

type MsgInfo struct {
	From    string
	To      []string
	Title   string
	Content string
}

type ServerInfo struct {
	Server   string
	Port     int
	Username string
	Passwd   string
}

func sendMail(msg MsgInfo, server ServerInfo) error {
	m := gomail.NewMessage()
	m.SetHeader("From", msg.From)
	m.SetHeader("To", msg.To...)
	m.SetHeader("Subject", msg.Title)
	m.SetBody("text/plain", msg.Content)

	d := gomail.NewDialer(server.Server, server.Port, server.Username, server.Passwd)

	return d.DialAndSend(m)
}
