package main

import (
	"fmt"

	"github.com/urfave/cli/v2"
	"github.com/urfave/cli/v2/altsrc"
)

var (
	configFileFlag = &cli.StringFlag{
		Name:    "config",
		Aliases: []string{"c"},
		Usage:   "The path to the config file",
		Value:   "config.toml",
	}

	debugFlag = &cli.BoolFlag{
		Name:  "debug",
		Usage: "Enable debug mode",
		Value: false,
	}

	dataDirFlag = &cli.StringFlag{
		Name:    "data-dir",
		Aliases: []string{"d"},
		Usage:   "The path to the data directory",
		Value:   "data",
	}

	metricsListenAddrFlag = &cli.StringFlag{
		Name:  "metrics-listen",
		Usage: "The listen address of the metrics server",
		Value: "0.0.0.0:20010",
	}

	apiListenAddrFlag = &cli.StringFlag{
		Name:  "api-listen",
		Usage: "The listen address of the api server",
		Value: "0.0.0.0:20012",
	}

	claimMonitorServerFlag = &cli.StringFlag{
		Name:  "claim-monitor-server",
		Usage: "The host of the claim monitor server",
	}

	privateKeyFlag = &cli.StringFlag{
		Name:    "private-key",
		Usage:   "The private key of the account",
		EnvVars: []string{"PRIVATE_KEY"},
	}

	chainRPCFlag = &cli.StringFlag{
		Name:  "chain-rpc",
		Usage: "The host of the chain json rpc server",
	}

	storeContractFlag = &cli.StringFlag{
		Name:  "store-contract",
		Usage: "The address of the store contract",
	}

	validatorContractFlag = &cli.StringFlag{
		Name:  "validator-contract",
		Usage: "The address of the reward contract",
	}

	commitOffsetFlag = &cli.IntFlag{
		Name:  "commit-offset",
		Usage: "The offset of the commit time, based on UTC, in seconds",
	}
)

var (
	questHostFlag = &cli.StringFlag{
		Name:  "quest-host",
		Usage: "The host of the quest server",
	}

	questPortFlag = &cli.IntFlag{
		Name:  "quest-port",
		Usage: "The port of the quest server",
	}

	questUserFlag = &cli.StringFlag{
		Name:  "quest-user",
		Usage: "The user of the quest server",
	}

	questPassFlag = &cli.StringFlag{
		Name:  "quest-pass",
		Usage: "The password of the quest server",
	}

	questDBFlag = &cli.StringFlag{
		Name:  "quest-db",
		Usage: "The database of the quest server",
	}
)

func loadFlagsFromConfig(cliCtx *cli.Context, flags []cli.Flag) error {
	if cliCtx.IsSet(configFileFlag.Name) {
		return altsrc.InitInputSourceWithContext(
			flags,
			altsrc.NewTomlSourceFromFlagFunc(configFileFlag.Name),
		)(cliCtx)
	}
	return nil
}

// WrapFlags so that they can be loaded from alternative sources.
func wrapFlags(flags []cli.Flag) []cli.Flag {
	wrapped := make([]cli.Flag, 0, len(flags))
	for _, f := range flags {
		switch t := f.(type) {
		case *cli.BoolFlag:
			f = altsrc.NewBoolFlag(t)
		case *cli.DurationFlag:
			f = altsrc.NewDurationFlag(t)
		case *cli.GenericFlag:
			f = altsrc.NewGenericFlag(t)
		case *cli.Float64Flag:
			f = altsrc.NewFloat64Flag(t)
		case *cli.IntFlag:
			f = altsrc.NewIntFlag(t)
		case *cli.StringFlag:
			f = altsrc.NewStringFlag(t)
		case *cli.StringSliceFlag:
			f = altsrc.NewStringSliceFlag(t)
		case *cli.Uint64Flag:
			f = altsrc.NewUint64Flag(t)
		case *cli.UintFlag:
			f = altsrc.NewUintFlag(t)
		case *cli.PathFlag:
			f = altsrc.NewPathFlag(t)
		case *cli.Int64Flag:
			// Int64Flag does not work. See https://github.com/prysmaticlabs/prysm/issues/6478
			panic(fmt.Sprintf("unsupported flag type type %T", f))
		case *cli.IntSliceFlag:
			f = altsrc.NewIntSliceFlag(t)
		default:
			panic(fmt.Sprintf("cannot convert type %T", f))
		}
		wrapped = append(wrapped, f)
	}
	return wrapped
}
