package main

import (
	"net/http"
	"os"
	"validator/api"
	"validator/claim_monitor"
	"validator/conf"
	"validator/core"
	"validator/quest"
	"validator/version"

	"github.com/prometheus/client_golang/prometheus/promhttp"
	log "github.com/sirupsen/logrus"
	"github.com/urfave/cli/v2"
)

var (
	appFlags = []cli.Flag{
		configFileFlag,
		dataDirFlag,
		metricsListenAddrFlag,
		debugFlag,
		apiListenAddrFlag,
		claimMonitorServerFlag,
		privateKeyFlag,
		chainRPCFlag,
		storeContractFlag,
		validatorContractFlag,
		commitOffsetFlag,
		questHostFlag,
		questPortFlag,
		questUserFlag,
		questPassFlag,
		questDBFlag,
	}
)

func main() {
	app := cli.App{}
	app.Flags = wrapFlags(append(appFlags, []cli.Flag{}...))
	app.Name = "validator"
	app.Usage = "this is validator"
	app.Version = version.Version
	app.Before = func(c *cli.Context) error {
		return loadFlagsFromConfig(c, app.Flags)
	}
	app.Action = func(context *cli.Context) error {
		run(context)
		return nil
	}
	if err := app.Run(os.Args); err != nil {
		log.Error(err)
	}
}

func run(ctx *cli.Context) {
	cfg := &conf.Config{
		MetricsListenAddr:  ctx.String(metricsListenAddrFlag.Name),
		Debug:              ctx.Bool(debugFlag.Name),
		APIListenAddr:      ctx.String(apiListenAddrFlag.Name),
		ClaimMonitorServer: ctx.String(claimMonitorServerFlag.Name),
		PrivateKey:         ctx.String(privateKeyFlag.Name),
		ChainRPC:           ctx.String(chainRPCFlag.Name),
		StoreContract:      ctx.String(storeContractFlag.Name),
		ValidatorContract:  ctx.String(validatorContractFlag.Name),
		DataDir:            ctx.String(dataDirFlag.Name),
		CommitOffset:       ctx.Int(commitOffsetFlag.Name),
	}

	if cfg.Debug {
		log.SetLevel(log.DebugLevel)
	}

	runMetrics(cfg.MetricsListenAddr)

	qCfg := &conf.QuestConfig{
		Host:     ctx.String(questHostFlag.Name),
		Port:     ctx.Int(questPortFlag.Name),
		User:     ctx.String(questUserFlag.Name),
		Password: ctx.String(questPassFlag.Name),
		Database: ctx.String(questDBFlag.Name),
		Debug:    cfg.Debug,
	}
	q := quest.NewQuest(qCfg)

	w := core.RunValidator(q, cfg)
	// runGrpcServer(cfg.GRPCListenAddr, w)
	monitorCli := claim_monitor.NewClient(cfg.ClaimMonitorServer)
	runAPIServer(cfg.APIListenAddr, w, monitorCli)
	select {}
}

func runMetrics(listen string) {
	http.Handle("/metrics", promhttp.Handler())
	log.WithField("listen", listen).Info("start prometheus metrics server")
	go func() {
		if err := http.ListenAndServe(listen, nil); err != nil {
			log.WithError(err).Error("failed to start prometheus metrics server")
		}
	}()
}

func runAPIServer(listen string, w *core.Validator, c *claim_monitor.Client) {
	go api.StartServer(listen, w, c)
}
