package peers

import (
	"context"
	"math"
	"math/rand"
	"time"

	"github.com/ethereum/go-ethereum/p2p/enr"
	"github.com/libp2p/go-libp2p/core/network"
	"github.com/libp2p/go-libp2p/core/peer"
	ma "github.com/multiformats/go-multiaddr"
	manet "github.com/multiformats/go-multiaddr/net"
	"github.com/prysmaticlabs/prysm/v3/beacon-chain/p2p/peers/peerdata"
)

const (
	// MinBackOffDuration minimum amount (in milliseconds) to wait before peer is re-dialed.
	// When node and peer are dialing each other simultaneously connection may fail. In order, to break
	// of constant dialing, peer is assigned some backoff period, and only dialed again once that backoff is up.
	MinBackOffDuration = 100
	// MaxBackOffDuration maximum amount (in milliseconds) to wait before peer is re-dialed.
	MaxBackOffDuration = 5000

	DefaultBadNodeReleaseDuration = time.Minute
)

type Status struct {
	ctx       context.Context
	store     *Store
	ipTracker map[string]uint64
	rand      *rand.Rand
	config    *StatusConfig
}

type StatusConfig struct {
	MaxInboundPeers        int
	MaxOutboundPeers       int
	MaxPeers               int
	MaxBadResponses        int
	BadNodeReleaseDuration time.Duration
}

func NewStatus(ctx context.Context, cfg *StatusConfig) *Status {
	store := NewStore(ctx, &storeConfig{
		MaxInboundPeers:  cfg.MaxInboundPeers,
		MaxOutboundPeers: cfg.MaxOutboundPeers,
		MaxPeers:         cfg.MaxPeers,
		MaxBadResponses:  cfg.MaxBadResponses,
	})

	// fallback set
	if cfg.BadNodeReleaseDuration == 0 {
		cfg.BadNodeReleaseDuration = DefaultBadNodeReleaseDuration
	}

	return &Status{
		ctx:       ctx,
		store:     store,
		ipTracker: make(map[string]uint64),
		rand:      rand.New(rand.NewSource(rand.Int63())),
		config:    cfg,
	}
}

// MaxPeerLimit returns the max peer limit stored in the current peer store.
func (p *Status) MaxPeerLimit() int {
	return p.store.Config().MaxPeers
}

// MaxInboundPeerLimit returns the max inbound peer limit stored in the current peer store.
func (p *Status) MaxInboundPeerLimit() int {
	return p.store.Config().MaxInboundPeers
}

// MaxOutboundPeerLimit returns the max inbound peer limit stored in the current peer store.
func (p *Status) MaxOutboundPeerLimit() int {
	return p.store.Config().MaxOutboundPeers
}

// MaxBadResponses returns the max bad responses stored in the current peer store.
func (p *Status) MaxBadResponses() int {
	return p.store.Config().MaxBadResponses
}

// Add adds a peer.
// If a peer already exists with this ID its address and direction are updated with the supplied data.
func (p *Status) Add(record *enr.Record, pid peer.ID, address ma.Multiaddr, direction network.Direction) {
	p.store.Lock()
	defer p.store.Unlock()

	if peerData, ok := p.store.PeerData(pid); ok {
		// Peer already exists, just update its address info.
		prevAddress := peerData.Address
		peerData.Address = address
		peerData.Direction = direction
		if record != nil {
			peerData.Enr = record
		}
		if !sameIP(prevAddress, address) {
			p.addIpToTracker(pid)
		}
		return
	}
	peerData := &PeerData{
		Address:   address,
		Direction: direction,
		// Peers start disconnected; state will be updated when the handshake process begins.
		ConnState: PeerDisconnected,
	}
	if record != nil {
		peerData.Enr = record
	}
	p.store.SetPeerData(pid, peerData)
	p.addIpToTracker(pid)
}

// SetConnectionState sets the connection state of the given remote peer.
func (p *Status) SetConnectionState(pid peer.ID, state PeerConnectionState) {
	p.store.Lock()
	defer p.store.Unlock()

	peerData := p.store.PeerDataGetOrCreate(pid)
	peerData.ConnState = state
}

// ConnectionState gets the connection state of the given remote peer.
// This will error if the peer does not exist.
func (p *Status) ConnectionState(pid peer.ID) (PeerConnectionState, error) {
	p.store.RLock()
	defer p.store.RUnlock()

	if peerData, ok := p.store.PeerData(pid); ok {
		return peerData.ConnState, nil
	}
	return PeerDisconnected, peerdata.ErrPeerUnknown
}

// InboundConnected returns the current batch of inbound peers that are connected.
func (p *Status) InboundConnected() []peer.ID {
	p.store.RLock()
	defer p.store.RUnlock()
	peers := make([]peer.ID, 0)
	for pid, peerData := range p.store.Peers() {
		if peerData.ConnState == PeerConnected && peerData.Direction == network.DirInbound {
			peers = append(peers, pid)
		}
	}
	return peers
}

// OutboundConnected returns the current batch of outbound peers that are connected.
func (p *Status) OutboundConnected() []peer.ID {
	p.store.RLock()
	defer p.store.RUnlock()
	peers := make([]peer.ID, 0)
	for pid, peerData := range p.store.Peers() {
		if peerData.ConnState == PeerConnected && peerData.Direction == network.DirOutbound {
			peers = append(peers, pid)
		}
	}
	return peers
}

// Active returns the peers that are connecting or connected.
func (p *Status) Active() []peer.ID {
	p.store.RLock()
	defer p.store.RUnlock()
	peers := make([]peer.ID, 0)
	for pid, peerData := range p.store.Peers() {
		if peerData.ConnState == PeerConnecting || peerData.ConnState == PeerConnected {
			peers = append(peers, pid)
		}
	}
	return peers
}

// RandomizeBackOff adds extra backoff period during which peer will not be dialed.
func (p *Status) RandomizeBackOff(pid peer.ID) {
	p.store.Lock()
	defer p.store.Unlock()

	peerData := p.store.PeerDataGetOrCreate(pid)

	// No need to add backoff period, if the previous one hasn't expired yet.
	if !time.Now().After(peerData.NextValidTime) {
		return
	}

	duration := time.Duration(math.Max(MinBackOffDuration, float64(p.rand.Intn(MaxBackOffDuration)))) * time.Millisecond
	peerData.NextValidTime = time.Now().Add(duration)
}

// IsReadyToDial checks where the given peer is ready to be
// dialed again.
func (p *Status) IsReadyToDial(pid peer.ID) bool {
	p.store.RLock()
	defer p.store.RUnlock()

	if peerData, ok := p.store.PeerData(pid); ok {
		timeIsZero := peerData.NextValidTime.IsZero()
		isInvalidTime := peerData.NextValidTime.After(time.Now())
		return timeIsZero || !isInvalidTime
	}
	// If no record exists, we don't restrict dials to the
	// peer.
	return true
}

// IsActive checks if a peers is active and returns the result appropriately.
func (p *Status) IsActive(pid peer.ID) bool {
	p.store.RLock()
	defer p.store.RUnlock()

	peerData, ok := p.store.PeerData(pid)
	return ok && (peerData.ConnState == PeerConnected || peerData.ConnState == PeerConnecting)
}

// IncBadResponses increments the number of bad responses received from the given peer.
func (p *Status) IncBadResponses(pid peer.ID) {
	p.store.Lock()
	defer p.store.Unlock()
	peerData, ok := p.store.PeerData(pid)
	if !ok {
		p.store.SetPeerData(pid, &PeerData{
			BadResponses: 1,
		})
		return
	}
	if time.Now().Before(peerData.NextBadNodeReleaseTime) {
		return
	}
	peerData.BadResponses++
	if peerData.BadResponses >= p.MaxBadResponses() {
		// freeze for a while
		peerData.NextBadNodeReleaseTime = time.Now().Add(p.config.BadNodeReleaseDuration)
	}
}

// IsBad states if the peer is to be considered bad (by *any* of the registered scorers).
// If the peer is unknown this will return `false`, which makes using this function easier than returning an error.
func (p *Status) IsBad(pid peer.ID) bool {
	p.store.RLock()
	defer p.store.RUnlock()
	return p.isBad(pid)
}

// isBad is the lock-free version of IsBad.
func (p *Status) isBad(pid peer.ID) bool {
	return p.isFromBadIP(pid) || p.isFromBadResponses(pid)
}

// this method assumes the store lock is acquired before
// executing the method.
func (p *Status) isFromBadIP(pid peer.ID) bool {
	peerData, ok := p.store.PeerData(pid)
	if !ok {
		return false
	}
	if peerData.Address == nil {
		return false
	}
	_, err := manet.ToIP(peerData.Address)
	if err != nil {
		return true
	}
	return false
}

// isFromBadResponses
func (p *Status) isFromBadResponses(pid peer.ID) bool {
	peerData, ok := p.store.PeerData(pid)
	if !ok {
		return false
	}

	// release bad node
	if !peerData.NextBadNodeReleaseTime.IsZero() && time.Now().After(peerData.NextBadNodeReleaseTime) {
		peerData.BadResponses = 0
		peerData.NextBadNodeReleaseTime = time.Time{}
	}
	return peerData.BadResponses >= p.MaxBadResponses()
}

func (p *Status) addIpToTracker(pid peer.ID) {
	data, ok := p.store.PeerData(pid)
	if !ok {
		return
	}
	if data.Address == nil {
		return
	}
	ip, err := manet.ToIP(data.Address)
	if err != nil {
		// Should never happen, it is
		// assumed every IP coming in
		// is a valid ip.
		return
	}
	// Ignore loopback addresses.
	if ip.IsLoopback() {
		return
	}
	stringIP := ip.String()
	p.ipTracker[stringIP] += 1
}

func (p *Status) tallyIPTracker() {
	tracker := map[string]uint64{}
	// Iterate through all peers.
	for _, peerData := range p.store.Peers() {
		if peerData.Address == nil {
			continue
		}
		ip, err := manet.ToIP(peerData.Address)
		if err != nil {
			// Should never happen, it is
			// assumed every IP coming in
			// is a valid ip.
			continue
		}
		stringIP := ip.String()
		tracker[stringIP] += 1
	}
	p.ipTracker = tracker
}

func sameIP(firstAddr, secondAddr ma.Multiaddr) bool {
	// Exit early if we do get nil multiaddresses
	if firstAddr == nil || secondAddr == nil {
		return false
	}
	firstIP, err := manet.ToIP(firstAddr)
	if err != nil {
		return false
	}
	secondIP, err := manet.ToIP(secondAddr)
	if err != nil {
		return false
	}
	return firstIP.Equal(secondIP)
}
